<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

// Check if user is logged in and is a company
if (!isLoggedIn() || !isCompany()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];

// Get company statistics
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT products.id) as total_products,
        COUNT(orders.id) as total_orders,
        SUM(CASE WHEN orders.status = 'Delivered' THEN 1 ELSE 0 END) as delivered_orders,
        SUM(CASE WHEN orders.status = 'Pending' THEN 1 ELSE 0 END) as pending_orders,
        SUM(CASE WHEN orders.payment_status = 'Paid' THEN orders.total_price ELSE 0 END) as total_revenue
    FROM products 
    LEFT JOIN orders ON products.id = orders.product_id 
    WHERE products.company_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Get recent orders - FIXED: Removed store_name reference
$recent_orders_stmt = $pdo->prepare("
    SELECT o.*, p.name as product_name, u.first_name, u.last_name
    FROM orders o 
    JOIN products p ON o.product_id = p.id 
    JOIN users u ON o.store_id = u.id 
    WHERE o.company_id = ? 
    ORDER BY o.created_at DESC 
    LIMIT 5
");
$recent_orders_stmt->execute([$user_id]);
$recent_orders = $recent_orders_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get low stock products
$low_stock_stmt = $pdo->prepare("
    SELECT * FROM products 
    WHERE company_id = ? AND stock < 10 
    ORDER BY stock ASC 
    LIMIT 5
");
$low_stock_stmt->execute([$user_id]);
$low_stock_products = $low_stock_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Company Dashboard - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            min-height: 100vh;
            position: fixed;
            width: 250px;
            transition: all 0.3s;
        }
        .sidebar .nav-link {
            color: white;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            transition: all 0.3s;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            padding-left: 30px;
        }
        .sidebar .nav-link i {
            width: 20px;
            margin-right: 10px;
        }
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .navbar-custom {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .stat-card {
            border: none;
            border-radius: 15px;
            transition: transform 0.3s, box-shadow 0.3s;
            margin-bottom: 20px;
        }
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        .stat-icon {
            font-size: 2.5rem;
            opacity: 0.7;
        }
        .bg-primary { background: linear-gradient(135deg, #667eea, #764ba2); }
        .bg-success { background: linear-gradient(135deg, #42b883, #47b8a4); }
        .bg-warning { background: linear-gradient(135deg, #ff9a44, #ff6b6b); }
        .bg-info { background: linear-gradient(135deg, #4facfe, #00f2fe); }
        .table-responsive { border-radius: 10px; }
        .badge-status { font-size: 0.8em; }
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
            }
            .sidebar.active {
                margin-left: 0;
            }
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-industry me-2"></i>Company Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link active" href="company_dashboard.php">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link" href="company_products.php">
                <i class="fas fa-boxes"></i>Products
            </a>
            <a class="nav-link" href="company_orders.php">
                <i class="fas fa-shopping-cart"></i>Orders
            </a>
            <a class="nav-link" href="company_transactions.php">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="company_messages.php">
                <i class="fas fa-comments"></i>Messages
            </a>
            <a class="nav-link" href="company_profile.php">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-custom mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-primary me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">Dashboard Overview</span>
                <div class="navbar-nav ms-auto">
                    <div class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-user-circle me-1"></i>
                            <?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="company_profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                        </ul>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Statistics Cards -->
        <div class="row">
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-primary stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['total_products'] ?? 0; ?></h4>
                                <p class="mb-0">Total Products</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-boxes"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-success stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['total_orders'] ?? 0; ?></h4>
                                <p class="mb-0">Total Orders</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-shopping-cart"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-warning stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['delivered_orders'] ?? 0; ?></h4>
                                <p class="mb-0">Delivered</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-check-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-info stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0">₦<?php echo number_format($stats['total_revenue'] ?? 0, 2); ?></h4>
                                <p class="mb-0">Total Revenue</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-money-bill-wave"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row mt-4">
            <!-- Recent Orders -->
            <div class="col-lg-8">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-clock me-2"></i>Recent Orders
                        </h5>
                        <a href="company_orders.php" class="btn btn-sm btn-primary">View All</a>
                    </div>
                    <div class="card-body">
                        <?php if ($recent_orders): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Store</th>
                                            <th>Qty</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_orders as $order): ?>
                                            <tr>
                                                <td><?php echo $order['product_name']; ?></td>
                                                <td><?php echo $order['first_name'] . ' ' . $order['last_name']; ?></td>
                                                <td><?php echo $order['quantity']; ?></td>
                                                <td>₦<?php echo number_format($order['total_price'], 2); ?></td>
                                                <td>
                                                    <span class="badge 
                                                        <?php 
                                                        switch($order['status']) {
                                                            case 'Delivered': echo 'bg-success'; break;
                                                            case 'Processing': echo 'bg-primary'; break;
                                                            case 'Pending': echo 'bg-warning'; break;
                                                            case 'Cancelled': echo 'bg-danger'; break;
                                                            default: echo 'bg-secondary';
                                                        }
                                                        ?> badge-status">
                                                        <?php echo $order['status']; ?>
                                                    </span>
                                                </td>
                                                <td><?php echo date('M j, Y', strtotime($order['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-4">
                                <i class="fas fa-shopping-cart fa-3x text-muted mb-3"></i>
                                <p class="text-muted">No orders yet</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Low Stock & Quick Actions -->
            <div class="col-lg-4">
                <!-- Low Stock Alert -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-exclamation-triangle me-2 text-warning"></i>Low Stock Alert
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if ($low_stock_products): ?>
                            <div class="list-group list-group-flush">
                                <?php foreach ($low_stock_products as $product): ?>
                                    <div class="list-group-item d-flex justify-content-between align-items-center">
                                        <div>
                                            <h6 class="mb-1"><?php echo $product['name']; ?></h6>
                                            <small class="text-muted">Stock: <?php echo $product['stock']; ?></small>
                                        </div>
                                        <a href="company_products.php?edit=<?php echo $product['id']; ?>" class="btn btn-sm btn-outline-warning">Restock</a>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-3">
                                <i class="fas fa-check-circle fa-2x text-success mb-2"></i>
                                <p class="text-muted mb-0">All products are well stocked</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>

                <!-- Quick Actions -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-bolt me-2"></i>Quick Actions
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="d-grid gap-2">
                            <a href="company_products.php?action=add" class="btn btn-primary">
                                <i class="fas fa-plus me-2"></i>Add New Product
                            </a>
                            <a href="company_orders.php" class="btn btn-outline-primary">
                                <i class="fas fa-eye me-2"></i>View All Orders
                            </a>
                            <a href="company_messages.php" class="btn btn-outline-primary">
                                <i class="fas fa-comments me-2"></i>Check Messages
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sidebar toggle for mobile
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });

        // Auto-hide sidebar on mobile when clicking outside
        document.addEventListener('click', function(event) {
            const sidebar = document.getElementById('sidebar');
            const sidebarToggle = document.getElementById('sidebarToggle');
            
            if (window.innerWidth <= 768 && 
                !sidebar.contains(event.target) && 
                !sidebarToggle.contains(event.target) &&
                sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                document.getElementById('sidebar').classList.remove('active');
            }
        });
    </script>
</body>
</html>