<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isCompany()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$message = '';

// Handle order status updates
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_status'])) {
        $order_id = intval($_POST['order_id']);
        $new_status = sanitize($_POST['status']);
        
        // Verify order belongs to company
        $verify_stmt = $pdo->prepare("SELECT id FROM orders WHERE id = ? AND company_id = ?");
        $verify_stmt->execute([$order_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $update_stmt = $pdo->prepare("UPDATE orders SET status = ? WHERE id = ?");
            if ($update_stmt->execute([$new_status, $order_id])) {
                $message = '<div class="alert alert-success">Order status updated successfully!</div>';
                
                // Create notification for store
                $store_stmt = $pdo->prepare("SELECT store_id FROM orders WHERE id = ?");
                $store_stmt->execute([$order_id]);
                $order = $store_stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($order) {
                    $notification_msg = "Your order #{$order_id} status has been updated to: {$new_status}";
                    createNotification($order['store_id'], $notification_msg, 'order', $order_id);
                }
            } else {
                $message = '<div class="alert alert-danger">Failed to update order status.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Order not found or access denied.</div>';
        }
    }
    
    // Handle payment confirmation
    if (isset($_POST['confirm_payment'])) {
        $order_id = intval($_POST['order_id']);
        
        // Verify order belongs to company
        $verify_stmt = $pdo->prepare("SELECT id FROM orders WHERE id = ? AND company_id = ?");
        $verify_stmt->execute([$order_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $update_stmt = $pdo->prepare("UPDATE orders SET payment_status = 'Paid' WHERE id = ?");
            if ($update_stmt->execute([$order_id])) {
                $message = '<div class="alert alert-success">Payment confirmed successfully!</div>';
            } else {
                $message = '<div class="alert alert-danger">Failed to confirm payment.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Order not found or access denied.</div>';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$payment_filter = $_GET['payment_status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build orders query with filters
$query = "
    SELECT o.*, 
           p.name as product_name,
           p.price as unit_price,
           u.first_name as store_first_name,
           u.last_name as store_last_name,
           u.phone as store_phone
    FROM orders o 
    JOIN products p ON o.product_id = p.id 
    JOIN users u ON o.store_id = u.id 
    WHERE o.company_id = ?
";

$params = [$user_id];

if ($status_filter) {
    $query .= " AND o.status = ?";
    $params[] = $status_filter;
}

if ($payment_filter) {
    $query .= " AND o.payment_status = ?";
    $params[] = $payment_filter;
}

if ($date_from) {
    $query .= " AND DATE(o.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $query .= " AND DATE(o.created_at) <= ?";
    $params[] = $date_to;
}

$query .= " ORDER BY o.created_at DESC";

$orders_stmt = $pdo->prepare($query);
$orders_stmt->execute($params);
$orders = $orders_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get order statistics
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_orders,
        SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending_orders,
        SUM(CASE WHEN status = 'Processing' THEN 1 ELSE 0 END) as processing_orders,
        SUM(CASE WHEN status = 'Delivered' THEN 1 ELSE 0 END) as delivered_orders,
        SUM(CASE WHEN payment_status = 'Paid' THEN total_price ELSE 0 END) as total_revenue,
        SUM(CASE WHEN payment_status = 'Pending' THEN total_price ELSE 0 END) as pending_revenue
    FROM orders 
    WHERE company_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Orders - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .order-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 20px;
        }
        .order-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .status-badge {
            font-size: 0.8em;
        }
        .order-actions {
            opacity: 0;
            transition: opacity 0.3s;
        }
        .order-card:hover .order-actions {
            opacity: 1;
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Company Sidebar -->
    <div class="sidebar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 100vh; position: fixed; width: 250px;">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-industry me-2"></i>Company Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link" href="company_dashboard.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link" href="company_products.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-boxes"></i>Products
            </a>
            <a class="nav-link active" href="company_orders.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1); background: rgba(255,255,255,0.1);">
                <i class="fas fa-shopping-cart"></i>Orders
            </a>
            <a class="nav-link" href="company_transactions.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="company_messages.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-comments"></i>Messages
            </a>
            <a class="nav-link" href="company_profile.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-primary me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-shopping-cart me-2"></i>Manage Orders
                </span>
            </div>
        </nav>

        <?php echo $message; ?>

        <!-- Order Statistics -->
        <div class="row mb-4">
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-primary">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['total_orders'] ?? 0; ?></h4>
                        <small>Total Orders</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-warning">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['pending_orders'] ?? 0; ?></h4>
                        <small>Pending</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-info">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['processing_orders'] ?? 0; ?></h4>
                        <small>Processing</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-success">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['delivered_orders'] ?? 0; ?></h4>
                        <small>Delivered</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-dark">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0">₦<?php echo number_format($stats['total_revenue'] ?? 0, 0); ?></h4>
                        <small>Revenue</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-secondary">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0">₦<?php echo number_format($stats['pending_revenue'] ?? 0, 0); ?></h4>
                        <small>Pending Revenue</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <h5 class="card-title">Filter Orders</h5>
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Order Status</label>
                        <select name="status" class="form-select">
                            <option value="">All Statuses</option>
                            <option value="Pending" <?php echo $status_filter === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="Processing" <?php echo $status_filter === 'Processing' ? 'selected' : ''; ?>>Processing</option>
                            <option value="Shipped" <?php echo $status_filter === 'Shipped' ? 'selected' : ''; ?>>Shipped</option>
                            <option value="Delivered" <?php echo $status_filter === 'Delivered' ? 'selected' : ''; ?>>Delivered</option>
                            <option value="Cancelled" <?php echo $status_filter === 'Cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Payment Status</label>
                        <select name="payment_status" class="form-select">
                            <option value="">All Payments</option>
                            <option value="Pending" <?php echo $payment_filter === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="Paid" <?php echo $payment_filter === 'Paid' ? 'selected' : ''; ?>>Paid</option>
                            <option value="Failed" <?php echo $payment_filter === 'Failed' ? 'selected' : ''; ?>>Failed</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">From Date</label>
                        <input type="date" name="date_from" class="form-control" value="<?php echo $date_from; ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">To Date</label>
                        <input type="date" name="date_to" class="form-control" value="<?php echo $date_to; ?>">
                    </div>
                    <div class="col-12">
                        <button type="submit" class="btn btn-primary">Apply Filters</button>
                        <a href="company_orders.php" class="btn btn-outline-secondary">Clear</a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Orders List -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">Order History</h5>
                <span class="badge bg-primary"><?php echo count($orders); ?> orders</span>
            </div>
            <div class="card-body">
                <?php if ($orders): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Order ID</th>
                                    <th>Product</th>
                                    <th>Store</th>
                                    <th>Quantity</th>
                                    <th>Amount</th>
                                    <th>Order Status</th>
                                    <th>Payment Status</th>
                                    <th>Date</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($orders as $order): ?>
                                    <tr>
                                        <td>
                                            <strong>#<?php echo $order['id']; ?></strong>
                                        </td>
                                        <td>
                                            <strong><?php echo $order['product_name']; ?></strong>
                                            <br>
                                            <small class="text-muted">₦<?php echo number_format($order['unit_price'], 2); ?> each</small>
                                        </td>
                                        <td>
                                            <?php echo $order['store_first_name'] . ' ' . $order['store_last_name']; ?>
                                            <br>
                                            <small class="text-muted"><?php echo $order['store_phone']; ?></small>
                                        </td>
                                        <td><?php echo $order['quantity']; ?></td>
                                        <td>
                                            <strong>₦<?php echo number_format($order['total_price'], 2); ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php 
                                                switch($order['status']) {
                                                    case 'Delivered': echo 'bg-success'; break;
                                                    case 'Processing': echo 'bg-primary'; break;
                                                    case 'Shipped': echo 'bg-info'; break;
                                                    case 'Pending': echo 'bg-warning'; break;
                                                    case 'Cancelled': echo 'bg-danger'; break;
                                                    default: echo 'bg-secondary';
                                                }
                                                ?> status-badge">
                                                <?php echo $order['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php 
                                                switch($order['payment_status']) {
                                                    case 'Paid': echo 'bg-success'; break;
                                                    case 'Pending': echo 'bg-warning'; break;
                                                    case 'Failed': echo 'bg-danger'; break;
                                                    default: echo 'bg-secondary';
                                                }
                                                ?> status-badge">
                                                <?php echo $order['payment_status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo date('M j, Y', strtotime($order['created_at'])); ?>
                                            <br>
                                            <small class="text-muted"><?php echo date('g:i A', strtotime($order['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <div class="order-actions">
                                                <!-- Status Update Dropdown -->
                                                <div class="dropdown d-inline-block">
                                                    <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                        <i class="fas fa-edit"></i>
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <li><h6 class="dropdown-header">Update Status</h6></li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                                <input type="hidden" name="status" value="Pending">
                                                                <button type="submit" name="update_status" class="btn btn-sm w-100 text-start">Pending</button>
                                                            </form>
                                                        </li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                                <input type="hidden" name="status" value="Processing">
                                                                <button type="submit" name="update_status" class="btn btn-sm w-100 text-start">Processing</button>
                                                            </form>
                                                        </li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                                <input type="hidden" name="status" value="Shipped">
                                                                <button type="submit" name="update_status" class="btn btn-sm w-100 text-start">Shipped</button>
                                                            </form>
                                                        </li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                                <input type="hidden" name="status" value="Delivered">
                                                                <button type="submit" name="update_status" class="btn btn-sm w-100 text-start">Delivered</button>
                                                            </form>
                                                        </li>
                                                    </ul>
                                                </div>

                                                <!-- Payment Confirmation -->
                                                <?php if ($order['payment_status'] === 'Pending'): ?>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" name="confirm_payment" class="btn btn-sm btn-outline-success" onclick="return confirm('Mark this payment as confirmed?')">
                                                            <i class="fas fa-check"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>

                                                <!-- Message Store -->
                                                <a href="company_messages.php?user_id=<?php echo $order['store_id']; ?>" class="btn btn-sm btn-outline-info">
                                                    <i class="fas fa-comment"></i>
                                                </a>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-shopping-cart fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Orders Yet</h4>
                        <p class="text-muted">Orders from stores will appear here.</p>
                        <?php if ($status_filter || $payment_filter || $date_from || $date_to): ?>
                            <a href="company_orders.php" class="btn btn-primary">Clear Filters</a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            const sidebar = document.querySelector('.sidebar');
            const mainContent = document.querySelector('.main-content');
            
            if (sidebar.style.marginLeft === '0px' || sidebar.style.marginLeft === '') {
                sidebar.style.marginLeft = '-250px';
                mainContent.style.marginLeft = '0';
            } else {
                sidebar.style.marginLeft = '0px';
                mainContent.style.marginLeft = '250px';
            }
        });

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);

        // Show order actions on hover
        document.querySelectorAll('tr').forEach(row => {
            row.addEventListener('mouseenter', function() {
                this.querySelector('.order-actions').style.opacity = '1';
            });
            row.addEventListener('mouseleave', function() {
                this.querySelector('.order-actions').style.opacity = '0';
            });
        });
    </script>
</body>
</html>