<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isCompany()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$message = '';

// Handle product actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['add_product'])) {
        $name = sanitize($_POST['name']);
        $category = sanitize($_POST['category']);
        $price = floatval($_POST['price']);
        $stock = intval($_POST['stock']);
        $description = sanitize($_POST['description']);

        // Validation
        $errors = [];
        if (empty($name)) {
            $errors[] = "Product name is required";
        }
        if (empty($category)) {
            $errors[] = "Category is required";
        }
        if ($price <= 0) {
            $errors[] = "Price must be greater than 0";
        }
        if ($stock < 0) {
            $errors[] = "Stock cannot be negative";
        }

        if (empty($errors)) {
            $stmt = $pdo->prepare("INSERT INTO products (company_id, name, category, price, stock, description) VALUES (?, ?, ?, ?, ?, ?)");
            if ($stmt->execute([$user_id, $name, $category, $price, $stock, $description])) {
                $message = '<div class="alert alert-success">Product added successfully!</div>';
                // Clear form
                $_POST = [];
            } else {
                $message = '<div class="alert alert-danger">Failed to add product. Please try again.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">' . implode('<br>', $errors) . '</div>';
        }
    }
    
    // Handle product deletion
    if (isset($_POST['delete_product'])) {
        $product_id = intval($_POST['product_id']);
        
        // Verify product belongs to company
        $verify_stmt = $pdo->prepare("SELECT id FROM products WHERE id = ? AND company_id = ?");
        $verify_stmt->execute([$product_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $delete_stmt = $pdo->prepare("DELETE FROM products WHERE id = ?");
            if ($delete_stmt->execute([$product_id])) {
                $message = '<div class="alert alert-success">Product deleted successfully!</div>';
            } else {
                $message = '<div class="alert alert-danger">Failed to delete product.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Product not found or access denied.</div>';
        }
    }
    
    // Handle product update
    if (isset($_POST['update_product'])) {
        $product_id = intval($_POST['product_id']);
        $name = sanitize($_POST['name']);
        $category = sanitize($_POST['category']);
        $price = floatval($_POST['price']);
        $stock = intval($_POST['stock']);
        $description = sanitize($_POST['description']);

        // Verify product belongs to company
        $verify_stmt = $pdo->prepare("SELECT id FROM products WHERE id = ? AND company_id = ?");
        $verify_stmt->execute([$product_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $update_stmt = $pdo->prepare("UPDATE products SET name = ?, category = ?, price = ?, stock = ?, description = ? WHERE id = ?");
            if ($update_stmt->execute([$name, $category, $price, $stock, $description, $product_id])) {
                $message = '<div class="alert alert-success">Product updated successfully!</div>';
            } else {
                $message = '<div class="alert alert-danger">Failed to update product.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Product not found or access denied.</div>';
        }
    }
}

// Get company products
$products_stmt = $pdo->prepare("SELECT * FROM products WHERE company_id = ? ORDER BY created_at DESC");
$products_stmt->execute([$user_id]);
$products = $products_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get product categories for dropdown
$categories_stmt = $pdo->prepare("SELECT DISTINCT category FROM products WHERE company_id = ? ORDER BY category");
$categories_stmt->execute([$user_id]);
$existing_categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Manage Products - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .product-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 20px;
        }
        .product-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        .low-stock {
            border-left: 4px solid #ffc107;
        }
        .out-of-stock {
            border-left: 4px solid #dc3545;
        }
        .in-stock {
            border-left: 4px solid #28a745;
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Company Sidebar -->
    <div class="sidebar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 100vh; position: fixed; width: 250px;">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-industry me-2"></i>Company Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link" href="company_dashboard.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link active" href="company_products.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1); background: rgba(255,255,255,0.1);">
                <i class="fas fa-boxes"></i>Products
            </a>
            <a class="nav-link" href="company_orders.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-shopping-cart"></i>Orders
            </a>
            <a class="nav-link" href="company_transactions.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="company_messages.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-comments"></i>Messages
            </a>
            <a class="nav-link" href="company_profile.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-primary me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-boxes me-2"></i>Manage Products
                </span>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addProductModal">
                    <i class="fas fa-plus me-2"></i>Add New Product
                </button>
            </div>
        </nav>

        <?php echo $message; ?>

        <!-- Products Summary -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card text-white bg-primary">
                    <div class="card-body text-center">
                        <h4 class="mb-0"><?php echo count($products); ?></h4>
                        <p class="mb-0">Total Products</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-white bg-success">
                    <div class="card-body text-center">
                        <h4 class="mb-0"><?php echo array_sum(array_column($products, 'stock')); ?></h4>
                        <p class="mb-0">Total Stock</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-white bg-warning">
                    <div class="card-body text-center">
                        <?php
                        $low_stock = array_filter($products, function($product) {
                            return $product['stock'] > 0 && $product['stock'] < 10;
                        });
                        ?>
                        <h4 class="mb-0"><?php echo count($low_stock); ?></h4>
                        <p class="mb-0">Low Stock</p>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card text-white bg-danger">
                    <div class="card-body text-center">
                        <?php
                        $out_of_stock = array_filter($products, function($product) {
                            return $product['stock'] == 0;
                        });
                        ?>
                        <h4 class="mb-0"><?php echo count($out_of_stock); ?></h4>
                        <p class="mb-0">Out of Stock</p>
                    </div>
                </div>
            </div>
        </div>

        <!-- Products List -->
        <div class="card">
            <div class="card-header">
                <h5 class="card-title mb-0">Your Products</h5>
            </div>
            <div class="card-body">
                <?php if ($products): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Product Name</th>
                                    <th>Category</th>
                                    <th>Price</th>
                                    <th>Stock</th>
                                    <th>Status</th>
                                    <th>Created</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($products as $product): ?>
                                    <tr>
                                        <td>
                                            <strong><?php echo $product['name']; ?></strong>
                                            <?php if ($product['description']): ?>
                                                <br><small class="text-muted"><?php echo substr($product['description'], 0, 50); ?>...</small>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo $product['category']; ?></td>
                                        <td>₦<?php echo number_format($product['price'], 2); ?></td>
                                        <td><?php echo $product['stock']; ?></td>
                                        <td>
                                            <?php if ($product['stock'] == 0): ?>
                                                <span class="badge bg-danger">Out of Stock</span>
                                            <?php elseif ($product['stock'] < 10): ?>
                                                <span class="badge bg-warning">Low Stock</span>
                                            <?php else: ?>
                                                <span class="badge bg-success">In Stock</span>
                                            <?php endif; ?>
                                        </td>
                                        <td><?php echo date('M j, Y', strtotime($product['created_at'])); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-primary" 
                                                    data-bs-toggle="modal" 
                                                    data-bs-target="#editProductModal"
                                                    onclick="editProduct(<?php echo $product['id']; ?>, '<?php echo addslashes($product['name']); ?>', '<?php echo addslashes($product['category']); ?>', <?php echo $product['price']; ?>, <?php echo $product['stock']; ?>, '<?php echo addslashes($product['description']); ?>')">
                                                <i class="fas fa-edit"></i>
                                            </button>
                                            <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to delete this product?')">
                                                <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                                                <button type="submit" name="delete_product" class="btn btn-sm btn-outline-danger">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            </form>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-box-open fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Products Yet</h4>
                        <p class="text-muted">Get started by adding your first product.</p>
                        <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addProductModal">
                            <i class="fas fa-plus me-2"></i>Add Your First Product
                        </button>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Add Product Modal -->
    <div class="modal fade" id="addProductModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Product</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Product Name *</label>
                                <input type="text" name="name" class="form-control" value="<?php echo $_POST['name'] ?? ''; ?>" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Category *</label>
                                <input type="text" name="category" class="form-control" list="categories" value="<?php echo $_POST['category'] ?? ''; ?>" required>
                                <datalist id="categories">
                                    <?php foreach ($existing_categories as $cat): ?>
                                        <option value="<?php echo $cat['category']; ?>">
                                    <?php endforeach; ?>
                                </datalist>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Price (₦) *</label>
                                <input type="number" name="price" class="form-control" step="0.01" min="0.01" value="<?php echo $_POST['price'] ?? ''; ?>" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Stock Quantity *</label>
                                <input type="number" name="stock" class="form-control" min="0" value="<?php echo $_POST['stock'] ?? '0'; ?>" required>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea name="description" class="form-control" rows="3" placeholder="Product description..."><?php echo $_POST['description'] ?? ''; ?></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="add_product" class="btn btn-primary">Add Product</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Product Modal -->
    <div class="modal fade" id="editProductModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Product</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <input type="hidden" name="product_id" id="edit_product_id">
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Product Name *</label>
                                <input type="text" name="name" id="edit_name" class="form-control" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Category *</label>
                                <input type="text" name="category" id="edit_category" class="form-control" list="categories" required>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Price (₦) *</label>
                                <input type="number" name="price" id="edit_price" class="form-control" step="0.01" min="0.01" required>
                            </div>
                            <div class="col-md-6 mb-3">
                                <label class="form-label">Stock Quantity *</label>
                                <input type="number" name="stock" id="edit_stock" class="form-control" min="0" required>
                            </div>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Description</label>
                            <textarea name="description" id="edit_description" class="form-control" rows="3"></textarea>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="update_product" class="btn btn-primary">Update Product</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            const sidebar = document.querySelector('.sidebar');
            const mainContent = document.querySelector('.main-content');
            
            if (sidebar.style.marginLeft === '0px' || sidebar.style.marginLeft === '') {
                sidebar.style.marginLeft = '-250px';
                mainContent.style.marginLeft = '0';
            } else {
                sidebar.style.marginLeft = '0px';
                mainContent.style.marginLeft = '250px';
            }
        });

        // Edit product function
        function editProduct(id, name, category, price, stock, description) {
            document.getElementById('edit_product_id').value = id;
            document.getElementById('edit_name').value = name;
            document.getElementById('edit_category').value = category;
            document.getElementById('edit_price').value = price;
            document.getElementById('edit_stock').value = stock;
            document.getElementById('edit_description').value = description;
        }

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);
    </script>
</body>
</html>