<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isCompany()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$message = '';

// Get current user data
$user_stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
$user_stmt->execute([$user_id]);
$user = $user_stmt->fetch(PDO::FETCH_ASSOC);

// Nigerian States and LGAs data
$nigerian_states = [
    'Abia' => ['Aba North', 'Aba South', 'Arochukwu', 'Bende', 'Ikawuno', 'Ikwuano', 'Isiala Ngwa North', 'Isiala Ngwa South', 'Isuikwuato', 'Umu Nneochi', 'Obi Ngwa', 'Osisoma Ngwa', 'Ugwunagbo', 'Ukwa East', 'Ukwa West', 'Umuahia North', 'Umuahia South'],
    'Adamawa' => ['Demsa', 'Fufore', 'Ganye', 'Girei', 'Gombi', 'Guyuk', 'Hong', 'Jada', 'Lamurde', 'Madagali', 'Maiha', 'Mayo Belwa', 'Michika', 'Mubi North', 'Mubi South', 'Numan', 'Shelleng', 'Song', 'Toungo', 'Yola North', 'Yola South'],
    'Lagos' => ['Agege', 'Ajeromi-Ifelodun', 'Alimosho', 'Amuwo-Odofin', 'Apapa', 'Badagry', 'Epe', 'Eti Osa', 'Ibeju-Lekki', 'Ifako-Ijaiye', 'Ikeja', 'Ikorodu', 'Kosofe', 'Lagos Island', 'Lagos Mainland', 'Mushin', 'Ojo', 'Oshodi-Isolo', 'Shomolu', 'Surulere'],
    'Rivers' => ['Port Harcourt', 'Obio-Akpor', 'Okrika', 'Ogu–Bolo', 'Eleme', 'Tai', 'Gokana', 'Khana', 'Oyigbo', 'Opobo–Nkoro', 'Andoni', 'Bonny', 'Degema', 'Asari-Toru', 'Akuku-Toru', 'Abua–Odual', 'Ahoada West', 'Ahoada East', 'Ogba–Egbema–Ndoni', 'Emohua', 'Ikwerre', 'Etche', 'Omuma']
];

// Handle profile updates
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_profile'])) {
        $first_name = sanitize($_POST['first_name']);
        $last_name = sanitize($_POST['last_name']);
        $other_name = sanitize($_POST['other_name']);
        $company_name = sanitize($_POST['company_name']);
        $email = sanitize($_POST['email']);
        $phone = sanitize($_POST['phone']);
        $state = sanitize($_POST['state']);
        $lga = sanitize($_POST['lga']);
        
        // Validation
        $errors = [];
        if (empty($first_name)) {
            $errors[] = "First name is required";
        }
        if (empty($last_name)) {
            $errors[] = "Last name is required";
        }
        if (empty($email) || !validateEmail($email)) {
            $errors[] = "Valid email is required";
        }
        if (empty($phone) || !validatePhone($phone)) {
            $errors[] = "Valid phone number is required";
        }
        if (empty($state) || !array_key_exists($state, $nigerian_states)) {
            $errors[] = "Please select a valid state";
        }
        if (empty($lga) || !in_array($lga, $nigerian_states[$state])) {
            $errors[] = "Please select a valid LGA";
        }

        // Check if email already exists (excluding current user)
        if (empty($errors)) {
            $email_check_stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? AND id != ?");
            $email_check_stmt->execute([$email, $user_id]);
            if ($email_check_stmt->fetch()) {
                $errors[] = "Email already exists";
            }
        }

        // Check if phone already exists (excluding current user)
        if (empty($errors)) {
            $phone_check_stmt = $pdo->prepare("SELECT id FROM users WHERE phone = ? AND id != ?");
            $phone_check_stmt->execute([$phone, $user_id]);
            if ($phone_check_stmt->fetch()) {
                $errors[] = "Phone number already exists";
            }
        }

        if (empty($errors)) {
            $update_stmt = $pdo->prepare("
                UPDATE users 
                SET first_name = ?, last_name = ?, other_name = ?, company_name = ?, email = ?, phone = ?, state = ?, lga = ? 
                WHERE id = ?
            ");
            
            if ($update_stmt->execute([$first_name, $last_name, $other_name, $company_name, $email, $phone, $state, $lga, $user_id])) {
                // Update session variables
                $_SESSION['first_name'] = $first_name;
                $_SESSION['last_name'] = $last_name;
                $_SESSION['email'] = $email;
                $_SESSION['state'] = $state;
                $_SESSION['lga'] = $lga;
                
                $message = '<div class="alert alert-success">Profile updated successfully!</div>';
                
                // Refresh user data
                $user_stmt->execute([$user_id]);
                $user = $user_stmt->fetch(PDO::FETCH_ASSOC);
            } else {
                $message = '<div class="alert alert-danger">Failed to update profile. Please try again.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">' . implode('<br>', $errors) . '</div>';
        }
    }
    
    // Handle password change
    if (isset($_POST['change_password'])) {
        $current_password = $_POST['current_password'];
        $new_password = $_POST['new_password'];
        $confirm_password = $_POST['confirm_password'];
        
        $errors = [];
        
        // Verify current password
        if (!password_verify($current_password, $user['password'])) {
            $errors[] = "Current password is incorrect";
        }
        
        if (empty($new_password)) {
            $errors[] = "New password is required";
        } elseif (strlen($new_password) < 6) {
            $errors[] = "New password must be at least 6 characters long";
        }
        
        if ($new_password !== $confirm_password) {
            $errors[] = "New passwords do not match";
        }
        
        if (empty($errors)) {
            $hashed_password = password_hash($new_password, PASSWORD_DEFAULT);
            $password_stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
            
            if ($password_stmt->execute([$hashed_password, $user_id])) {
                $message = '<div class="alert alert-success">Password changed successfully!</div>';
            } else {
                $message = '<div class="alert alert-danger">Failed to change password. Please try again.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">' . implode('<br>', $errors) . '</div>';
        }
    }
}

// Get company statistics for profile
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(DISTINCT products.id) as total_products,
        COUNT(orders.id) as total_orders,
        SUM(CASE WHEN orders.status = 'Delivered' THEN 1 ELSE 0 END) as delivered_orders,
        SUM(CASE WHEN orders.payment_status = 'Paid' THEN orders.total_price ELSE 0 END) as total_revenue
    FROM products 
    LEFT JOIN orders ON products.id = orders.product_id 
    WHERE products.company_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Company Profile - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .profile-card {
            border: none;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .profile-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 15px 15px 0 0;
            padding: 30px;
            text-align: center;
        }
        .profile-avatar {
            width: 100px;
            height: 100px;
            border-radius: 50%;
            background: rgba(255,255,255,0.2);
            display: flex;
            align-items: center;
            justify-content: center;
            margin: 0 auto 15px;
            font-size: 2.5rem;
        }
        .stat-card {
            border: none;
            border-radius: 10px;
            transition: transform 0.3s;
        }
        .stat-card:hover {
            transform: translateY(-5px);
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Company Sidebar -->
    <div class="sidebar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 100vh; position: fixed; width: 250px;">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-industry me-2"></i>Company Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link" href="company_dashboard.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link" href="company_products.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-boxes"></i>Products
            </a>
            <a class="nav-link" href="company_orders.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-shopping-cart"></i>Orders
            </a>
            <a class="nav-link" href="company_transactions.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="company_messages.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-comments"></i>Messages
            </a>
            <a class="nav-link active" href="company_profile.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1); background: rgba(255,255,255,0.1);">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-primary me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-user-cog me-2"></i>Company Profile
                </span>
            </div>
        </nav>

        <?php echo $message; ?>

        <div class="row">
            <!-- Profile Information -->
            <div class="col-lg-8">
                <div class="card profile-card mb-4">
                    <div class="profile-header">
                        <div class="profile-avatar">
                            <i class="fas fa-building"></i>
                        </div>
                        <h3><?php echo $user['company_name'] ?? $user['first_name'] . ' ' . $user['last_name']; ?></h3>
                        <p class="mb-0">Company Account</p>
                    </div>
                    <div class="card-body">
                        <!-- Profile Update Form -->
                        <form method="POST">
                            <h5 class="card-title mb-4">
                                <i class="fas fa-user-edit me-2"></i>Profile Information
                            </h5>
                            
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">First Name *</label>
                                    <input type="text" name="first_name" class="form-control" 
                                           value="<?php echo $user['first_name']; ?>" required>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Last Name *</label>
                                    <input type="text" name="last_name" class="form-control" 
                                           value="<?php echo $user['last_name']; ?>" required>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Other Name</label>
                                    <input type="text" name="other_name" class="form-control" 
                                           value="<?php echo $user['other_name'] ?? ''; ?>">
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Company Name</label>
                                    <input type="text" name="company_name" class="form-control" 
                                           value="<?php echo $user['company_name'] ?? ''; ?>" 
                                           placeholder="Your company name (optional)">
                                </div>
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Email Address *</label>
                                    <input type="email" name="email" class="form-control" 
                                           value="<?php echo $user['email']; ?>" required>
                                </div>
                            </div>

                            <div class="row">
                                <div class="col-md-6 mb-3">
                                    <label class="form-label">Phone Number *</label>
                                    <input type="tel" name="phone" class="form-control" 
                                           value="<?php echo $user['phone']; ?>" required>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">State *</label>
                                    <select name="state" class="form-select" id="state" required>
                                        <option value="">Select State</option>
                                        <?php foreach ($nigerian_states as $state_name => $lgas): ?>
                                            <option value="<?php echo $state_name; ?>" 
                                                <?php echo ($user['state'] === $state_name) ? 'selected' : ''; ?>>
                                                <?php echo $state_name; ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="col-md-3 mb-3">
                                    <label class="form-label">LGA *</label>
                                    <select name="lga" class="form-select" id="lga" required>
                                        <option value="">Select LGA</option>
                                        <?php if (!empty($user['state'])): ?>
                                            <?php foreach ($nigerian_states[$user['state']] as $lga_name): ?>
                                                <option value="<?php echo $lga_name; ?>"
                                                    <?php echo ($user['lga'] === $lga_name) ? 'selected' : ''; ?>>
                                                    <?php echo $lga_name; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                </div>
                            </div>

                            <div class="mb-3">
                                <label class="form-label">Account Created</label>
                                <input type="text" class="form-control" 
                                       value="<?php echo date('F j, Y g:i A', strtotime($user['created_at'])); ?>" 
                                       readonly style="background-color: #f8f9fa;">
                            </div>

                            <div class="d-grid">
                                <button type="submit" name="update_profile" class="btn btn-primary btn-lg">
                                    <i class="fas fa-save me-2"></i>Update Profile
                                </button>
                            </div>
                        </form>
                    </div>
                </div>

                <!-- Change Password -->
                <div class="card profile-card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">
                            <i class="fas fa-lock me-2"></i>Change Password
                        </h5>
                        
                        <form method="POST">
                            <div class="row">
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Current Password *</label>
                                    <input type="password" name="current_password" class="form-control" required>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">New Password *</label>
                                    <input type="password" name="new_password" class="form-control" required>
                                    <div class="form-text">Minimum 6 characters</div>
                                </div>
                                <div class="col-md-4 mb-3">
                                    <label class="form-label">Confirm New Password *</label>
                                    <input type="password" name="confirm_password" class="form-control" required>
                                </div>
                            </div>
                            
                            <div class="d-grid">
                                <button type="submit" name="change_password" class="btn btn-warning">
                                    <i class="fas fa-key me-2"></i>Change Password
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Company Statistics -->
            <div class="col-lg-4">
                <!-- Quick Stats -->
                <div class="card profile-card mb-4">
                    <div class="card-body">
                        <h5 class="card-title mb-4">
                            <i class="fas fa-chart-bar me-2"></i>Business Overview
                        </h5>
                        
                        <div class="row text-center">
                            <div class="col-6 mb-4">
                                <div class="stat-card bg-light p-3 rounded">
                                    <i class="fas fa-box fa-2x text-primary mb-2"></i>
                                    <h4 class="mb-0"><?php echo $stats['total_products'] ?? 0; ?></h4>
                                    <small>Total Products</small>
                                </div>
                            </div>
                            <div class="col-6 mb-4">
                                <div class="stat-card bg-light p-3 rounded">
                                    <i class="fas fa-shopping-cart fa-2x text-success mb-2"></i>
                                    <h4 class="mb-0"><?php echo $stats['total_orders'] ?? 0; ?></h4>
                                    <small>Total Orders</small>
                                </div>
                            </div>
                            <div class="col-6 mb-4">
                                <div class="stat-card bg-light p-3 rounded">
                                    <i class="fas fa-check-circle fa-2x text-info mb-2"></i>
                                    <h4 class="mb-0"><?php echo $stats['delivered_orders'] ?? 0; ?></h4>
                                    <small>Delivered</small>
                                </div>
                            </div>
                            <div class="col-6 mb-4">
                                <div class="stat-card bg-light p-3 rounded">
                                    <i class="fas fa-money-bill-wave fa-2x text-warning mb-2"></i>
                                    <h4 class="mb-0">₦<?php echo number_format($stats['total_revenue'] ?? 0, 0); ?></h4>
                                    <small>Revenue</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Account Information -->
                <div class="card profile-card">
                    <div class="card-body">
                        <h5 class="card-title mb-4">
                            <i class="fas fa-info-circle me-2"></i>Account Information
                        </h5>
                        
                        <div class="mb-3">
                            <small class="text-muted">Account Type</small>
                            <p class="mb-0">
                                <i class="fas fa-building me-2 text-primary"></i>
                                <strong>Company Account</strong>
                            </p>
                        </div>
                        
                        <div class="mb-3">
                            <small class="text-muted">Member Since</small>
                            <p class="mb-0">
                                <i class="fas fa-calendar me-2 text-success"></i>
                                <?php echo date('F Y', strtotime($user['created_at'])); ?>
                            </p>
                        </div>
                        
                        <div class="mb-3">
                            <small class="text-muted">Location</small>
                            <p class="mb-0">
                                <i class="fas fa-map-marker-alt me-2 text-danger"></i>
                                <?php echo $user['state'] . ', ' . $user['lga']; ?>
                            </p>
                        </div>
                        
                        <div class="mb-0">
                            <small class="text-muted">Last Login</small>
                            <p class="mb-0">
                                <i class="fas fa-clock me-2 text-info"></i>
                                Just now
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            const sidebar = document.querySelector('.sidebar');
            const mainContent = document.querySelector('.main-content');
            
            if (sidebar.style.marginLeft === '0px' || sidebar.style.marginLeft === '') {
                sidebar.style.marginLeft = '-250px';
                mainContent.style.marginLeft = '0';
            } else {
                sidebar.style.marginLeft = '0px';
                mainContent.style.marginLeft = '250px';
            }
        });

        // Dynamic LGA loading based on state selection
        document.getElementById('state').addEventListener('change', function() {
            const state = this.value;
            const lgaSelect = document.getElementById('lga');
            
            // Nigerian States and LGAs data
            const statesLgas = <?php echo json_encode($nigerian_states); ?>;
            
            // Clear existing options
            lgaSelect.innerHTML = '<option value="">Select LGA</option>';
            
            if (state && statesLgas[state]) {
                statesLgas[state].forEach(lga => {
                    const option = document.createElement('option');
                    option.value = lga;
                    option.textContent = lga;
                    lgaSelect.appendChild(option);
                });
            }
        });

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);

        // Phone number formatting
        document.querySelector('input[name="phone"]')?.addEventListener('input', function(e) {
            const value = e.target.value.replace(/\D/g, '');
            
            if (value.length <= 3) {
                e.target.value = value;
            } else if (value.length <= 6) {
                e.target.value = value.slice(0, 3) + ' ' + value.slice(3);
            } else {
                e.target.value = value.slice(0, 3) + ' ' + value.slice(3, 6) + ' ' + value.slice(6, 11);
            }
        });
    </script>
</body>
</html>