<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isCompany()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$message = '';

// Handle transaction actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['update_transaction'])) {
        $transaction_id = intval($_POST['transaction_id']);
        $new_status = sanitize($_POST['status']);
        
        // Verify transaction belongs to company
        $verify_stmt = $pdo->prepare("
            SELECT t.id 
            FROM transactions t 
            JOIN orders o ON t.order_id = o.id 
            WHERE t.id = ? AND o.company_id = ?
        ");
        $verify_stmt->execute([$transaction_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $update_stmt = $pdo->prepare("UPDATE transactions SET status = ? WHERE id = ?");
            if ($update_stmt->execute([$new_status, $transaction_id])) {
                $message = '<div class="alert alert-success">Transaction status updated successfully!</div>';
                
                // Update corresponding order payment status
                if ($new_status === 'Completed') {
                    $order_stmt = $pdo->prepare("
                        UPDATE orders o 
                        JOIN transactions t ON o.id = t.order_id 
                        SET o.payment_status = 'Paid' 
                        WHERE t.id = ?
                    ");
                    $order_stmt->execute([$transaction_id]);
                }
            } else {
                $message = '<div class="alert alert-danger">Failed to update transaction status.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Transaction not found or access denied.</div>';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$payment_method = $_GET['payment_method'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';

// Build transactions query with filters - FIXED: Removed payment_gateway reference
$query = "
    SELECT t.*, 
           o.id as order_id,
           o.quantity,
           o.total_price as order_amount,
           o.status as order_status,
           o.payment_status as order_payment_status,
           p.name as product_name,
           u.first_name as store_first_name,
           u.last_name as store_last_name,
           u.phone as store_phone
    FROM transactions t 
    JOIN orders o ON t.order_id = o.id 
    JOIN products p ON o.product_id = p.id 
    JOIN users u ON o.store_id = u.id 
    WHERE o.company_id = ?
";

$params = [$user_id];

if ($status_filter) {
    $query .= " AND t.status = ?";
    $params[] = $status_filter;
}

if ($payment_method) {
    $query .= " AND t.payment_method = ?";
    $params[] = $payment_method;
}

if ($date_from) {
    $query .= " AND DATE(t.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $query .= " AND DATE(t.created_at) <= ?";
    $params[] = $date_to;
}

$query .= " ORDER BY t.created_at DESC";

$transactions_stmt = $pdo->prepare($query);
$transactions_stmt->execute($params);
$transactions = $transactions_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get transaction statistics - FIXED: Removed payment_gateway references
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_transactions,
        SUM(CASE WHEN t.status = 'Completed' THEN t.amount ELSE 0 END) as total_received,
        SUM(CASE WHEN t.status = 'Pending' THEN t.amount ELSE 0 END) as pending_amount,
        SUM(CASE WHEN t.status = 'Failed' THEN t.amount ELSE 0 END) as failed_amount,
        COUNT(CASE WHEN t.payment_method = 'Card' THEN 1 END) as card_payments,
        COUNT(CASE WHEN t.payment_method = 'Transfer' THEN 1 END) as transfer_payments,
        COUNT(CASE WHEN t.payment_method = 'Cash' THEN 1 END) as cash_payments
    FROM transactions t 
    JOIN orders o ON t.order_id = o.id 
    WHERE o.company_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Get recent transactions for dashboard (last 7 days)
$recent_stats_stmt = $pdo->prepare("
    SELECT 
        DATE(t.created_at) as date,
        COUNT(*) as transaction_count,
        SUM(t.amount) as daily_revenue
    FROM transactions t 
    JOIN orders o ON t.order_id = o.id 
    WHERE o.company_id = ? AND t.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    GROUP BY DATE(t.created_at)
    ORDER BY date DESC
");
$recent_stats_stmt->execute([$user_id]);
$recent_stats = $recent_stats_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Transactions - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .transaction-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 20px;
        }
        .transaction-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .status-badge {
            font-size: 0.8em;
        }
        .amount-positive {
            color: #28a745;
            font-weight: bold;
        }
        .revenue-chart {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            border-radius: 10px;
            padding: 20px;
            color: white;
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Company Sidebar -->
    <div class="sidebar" style="background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; min-height: 100vh; position: fixed; width: 250px;">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-industry me-2"></i>Company Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link" href="company_dashboard.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link" href="company_products.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-boxes"></i>Products
            </a>
            <a class="nav-link" href="company_orders.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-shopping-cart"></i>Orders
            </a>
            <a class="nav-link active" href="company_transactions.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1); background: rgba(255,255,255,0.1);">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="company_messages.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-comments"></i>Messages
            </a>
            <a class="nav-link" href="company_profile.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php" style="color: white; padding: 15px 20px; border-bottom: 1px solid rgba(255,255,255,0.1);">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-primary me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-money-bill-wave me-2"></i>Transactions
                </span>
            </div>
        </nav>

        <?php echo $message; ?>

        <!-- Transaction Statistics -->
        <div class="row mb-4">
            <div class="col-xl-3 col-md-6 mb-3">
                <div class="card text-white bg-primary">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0">₦<?php echo number_format($stats['total_received'] ?? 0, 0); ?></h4>
                                <p class="mb-0">Total Received</p>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-wallet fa-2x"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6 mb-3">
                <div class="card text-white bg-warning">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0">₦<?php echo number_format($stats['pending_amount'] ?? 0, 0); ?></h4>
                                <p class="mb-0">Pending</p>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-clock fa-2x"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6 mb-3">
                <div class="card text-white bg-success">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['total_transactions'] ?? 0; ?></h4>
                                <p class="mb-0">Total Transactions</p>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-exchange-alt fa-2x"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6 mb-3">
                <div class="card text-white bg-danger">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0">₦<?php echo number_format($stats['failed_amount'] ?? 0, 0); ?></h4>
                                <p class="mb-0">Failed</p>
                            </div>
                            <div class="align-self-center">
                                <i class="fas fa-times-circle fa-2x"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Payment Method Statistics -->
        <div class="row mb-4">
            <div class="col-md-6">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">Payment Methods</h5>
                    </div>
                    <div class="card-body">
                        <div class="row text-center">
                            <div class="col-4">
                                <div class="border rounded p-3">
                                    <i class="fas fa-credit-card fa-2x text-primary mb-2"></i>
                                    <h5><?php echo $stats['card_payments'] ?? 0; ?></h5>
                                    <small>Card Payments</small>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="border rounded p-3">
                                    <i class="fas fa-university fa-2x text-success mb-2"></i>
                                    <h5><?php echo $stats['transfer_payments'] ?? 0; ?></h5>
                                    <small>Transfers</small>
                                </div>
                            </div>
                            <div class="col-4">
                                <div class="border rounded p-3">
                                    <i class="fas fa-money-bill fa-2x text-info mb-2"></i>
                                    <h5><?php echo $stats['cash_payments'] ?? 0; ?></h5>
                                    <small>Cash</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-md-6">
                <div class="card revenue-chart">
                    <div class="card-body">
                        <h5 class="card-title text-white">Recent Revenue (7 Days)</h5>
                        <?php if ($recent_stats): ?>
                            <div class="mt-3">
                                <?php foreach ($recent_stats as $day): ?>
                                    <div class="d-flex justify-content-between text-white mb-2">
                                        <span><?php echo date('D, M j', strtotime($day['date'])); ?></span>
                                        <span>₦<?php echo number_format($day['daily_revenue'], 0); ?></span>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php else: ?>
                            <p class="text-white-50">No transactions in the last 7 days</p>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <h5 class="card-title">Filter Transactions</h5>
                <form method="GET" class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Transaction Status</label>
                        <select name="status" class="form-select">
                            <option value="">All Statuses</option>
                            <option value="Pending" <?php echo $status_filter === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="Completed" <?php echo $status_filter === 'Completed' ? 'selected' : ''; ?>>Completed</option>
                            <option value="Failed" <?php echo $status_filter === 'Failed' ? 'selected' : ''; ?>>Failed</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Payment Method</label>
                        <select name="payment_method" class="form-select">
                            <option value="">All Methods</option>
                            <option value="Card" <?php echo $payment_method === 'Card' ? 'selected' : ''; ?>>Card</option>
                            <option value="Transfer" <?php echo $payment_method === 'Transfer' ? 'selected' : ''; ?>>Transfer</option>
                            <option value="Cash" <?php echo $payment_method === 'Cash' ? 'selected' : ''; ?>>Cash</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">From Date</label>
                        <input type="date" name="date_from" class="form-control" value="<?php echo $date_from; ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">To Date</label>
                        <input type="date" name="date_to" class="form-control" value="<?php echo $date_to; ?>">
                    </div>
                    <div class="col-12">
                        <button type="submit" class="btn btn-primary">Apply Filters</button>
                        <a href="company_transactions.php" class="btn btn-outline-secondary">Clear</a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Transactions List -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">Transaction History</h5>
                <span class="badge bg-primary"><?php echo count($transactions); ?> transactions</span>
            </div>
            <div class="card-body">
                <?php if ($transactions): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Transaction ID</th>
                                    <th>Order ID</th>
                                    <th>Store</th>
                                    <th>Product</th>
                                    <th>Amount</th>
                                    <th>Payment Method</th>
                                    <th>Status</th>
                                    <th>Date</th>
                                    <th>Reference</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($transactions as $transaction): ?>
                                    <tr>
                                        <td>
                                            <strong>#<?php echo $transaction['id']; ?></strong>
                                        </td>
                                        <td>
                                            <a href="company_orders.php" class="text-decoration-none">
                                                #<?php echo $transaction['order_id']; ?>
                                            </a>
                                        </td>
                                        <td>
                                            <?php echo $transaction['store_first_name'] . ' ' . $transaction['store_last_name']; ?>
                                            <br>
                                            <small class="text-muted"><?php echo $transaction['store_phone']; ?></small>
                                        </td>
                                        <td>
                                            <strong><?php echo $transaction['product_name']; ?></strong>
                                            <br>
                                            <small class="text-muted">Qty: <?php echo $transaction['quantity']; ?></small>
                                        </td>
                                        <td>
                                            <span class="amount-positive">₦<?php echo number_format($transaction['amount'], 2); ?></span>
                                        </td>
                                        <td>
                                            <span class="badge bg-secondary"><?php echo $transaction['payment_method']; ?></span>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php 
                                                switch($transaction['status']) {
                                                    case 'Completed': echo 'bg-success'; break;
                                                    case 'Pending': echo 'bg-warning'; break;
                                                    case 'Failed': echo 'bg-danger'; break;
                                                    default: echo 'bg-secondary';
                                                }
                                                ?> status-badge">
                                                <?php echo $transaction['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo date('M j, Y', strtotime($transaction['created_at'])); ?>
                                            <br>
                                            <small class="text-muted"><?php echo date('g:i A', strtotime($transaction['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <small class="text-muted"><?php echo $transaction['transaction_ref'] ?: 'N/A'; ?></small>
                                        </td>
                                        <td>
                                            <div class="btn-group">
                                                <!-- Status Update Dropdown -->
                                                <div class="dropdown">
                                                    <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                                                        <i class="fas fa-cog"></i>
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <li><h6 class="dropdown-header">Update Status</h6></li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="transaction_id" value="<?php echo $transaction['id']; ?>">
                                                                <input type="hidden" name="status" value="Pending">
                                                                <button type="submit" name="update_transaction" class="btn btn-sm w-100 text-start">Mark as Pending</button>
                                                            </form>
                                                        </li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="transaction_id" value="<?php echo $transaction['id']; ?>">
                                                                <input type="hidden" name="status" value="Completed">
                                                                <button type="submit" name="update_transaction" class="btn btn-sm w-100 text-start">Mark as Completed</button>
                                                            </form>
                                                        </li>
                                                        <li>
                                                            <form method="POST" class="dropdown-item p-0">
                                                                <input type="hidden" name="transaction_id" value="<?php echo $transaction['id']; ?>">
                                                                <input type="hidden" name="status" value="Failed">
                                                                <button type="submit" name="update_transaction" class="btn btn-sm w-100 text-start">Mark as Failed</button>
                                                            </form>
                                                        </li>
                                                    </ul>
                                                </div>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-money-bill-wave fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Transactions Yet</h4>
                        <p class="text-muted">Transactions will appear here when stores make payments.</p>
                        <?php if ($status_filter || $payment_method || $date_from || $date_to): ?>
                            <a href="company_transactions.php" class="btn btn-primary">Clear Filters</a>
                        <?php endif; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            const sidebar = document.querySelector('.sidebar');
            const mainContent = document.querySelector('.main-content');
            
            if (sidebar.style.marginLeft === '0px' || sidebar.style.marginLeft === '') {
                sidebar.style.marginLeft = '-250px';
                mainContent.style.marginLeft = '0';
            } else {
                sidebar.style.marginLeft = '0px';
                mainContent.style.marginLeft = '250px';
            }
        });

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);
    </script>
</body>
</html>