<?php
// Message-related functions
function sendMessage($sender_id, $receiver_id, $message, $subject = '', $message_type = 'General', $parent_id = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO messages (sender_id, receiver_id, subject, message, message_type, parent_id) 
            VALUES (?, ?, ?, ?, ?, ?)
        ");
        
        $result = $stmt->execute([$sender_id, $receiver_id, $subject, $message, $message_type, $parent_id]);
        
        if ($result) {
            // Create notification for receiver
            createNotification($receiver_id, "New message from " . getUserName($sender_id), 'message', $pdo->lastInsertId());
            return true;
        }
    } catch (PDOException $e) {
        error_log("Message sending error: " . $e->getMessage());
        return false;
    }
    
    return false;
}

function getConversation($user1_id, $user2_id, $limit = 50) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT m.*, 
                   sender.first_name as sender_first_name,
                   sender.last_name as sender_last_name,
                   receiver.first_name as receiver_first_name,
                   receiver.last_name as receiver_last_name
            FROM messages m
            JOIN users sender ON m.sender_id = sender.id
            JOIN users receiver ON m.receiver_id = receiver.id
            WHERE (m.sender_id = ? AND m.receiver_id = ?) 
               OR (m.sender_id = ? AND m.receiver_id = ?)
            ORDER BY m.created_at ASC
            LIMIT ?
        ");
        
        $stmt->execute([$user1_id, $user2_id, $user2_id, $user1_id, $limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Get conversation error: " . $e->getMessage());
        return [];
    }
}

function getMessageThreads($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT 
                u.id as other_user_id,
                u.first_name,
                u.last_name,
                u.role,
                u.company_name,
                u.store_name,
                m.message,
                m.created_at,
                m.is_read,
                (SELECT COUNT(*) FROM messages WHERE receiver_id = ? AND sender_id = u.id AND is_read = FALSE) as unread_count
            FROM users u
            INNER JOIN messages m ON (
                (m.sender_id = u.id AND m.receiver_id = ?) OR 
                (m.sender_id = ? AND m.receiver_id = u.id)
            )
            WHERE u.id IN (
                SELECT DISTINCT 
                    CASE 
                        WHEN sender_id = ? THEN receiver_id 
                        ELSE sender_id 
                    END as other_user_id
                FROM messages 
                WHERE sender_id = ? OR receiver_id = ?
            )
            GROUP BY u.id, u.first_name, u.last_name, u.role, u.company_name, u.store_name
            ORDER BY MAX(m.created_at) DESC
        ");
        
        $stmt->execute([$user_id, $user_id, $user_id, $user_id, $user_id, $user_id]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("Get message threads error: " . $e->getMessage());
        return [];
    }
}

function markMessagesAsRead($user1_id, $user2_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            UPDATE messages 
            SET is_read = TRUE 
            WHERE sender_id = ? AND receiver_id = ? AND is_read = FALSE
        ");
        
        return $stmt->execute([$user2_id, $user1_id]);
    } catch (PDOException $e) {
        error_log("Mark messages as read error: " . $e->getMessage());
        return false;
    }
}

function getUnreadMessageCount($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            SELECT COUNT(*) as unread_count 
            FROM messages 
            WHERE receiver_id = ? AND is_read = FALSE
        ");
        
        $stmt->execute([$user_id]);
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['unread_count'] ?? 0;
    } catch (PDOException $e) {
        error_log("Get unread count error: " . $e->getMessage());
        return 0;
    }
}

function createNotification($user_id, $message, $type = 'system', $related_id = null) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("
            INSERT INTO notifications (user_id, message, type, related_id) 
            VALUES (?, ?, ?, ?)
        ");
        
        return $stmt->execute([$user_id, $message, $type, $related_id]);
    } catch (PDOException $e) {
        error_log("Create notification error: " . $e->getMessage());
        return false;
    }
}

function getUserName($user_id) {
    global $pdo;
    
    try {
        $stmt = $pdo->prepare("SELECT first_name, last_name FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return $user ? $user['first_name'] . ' ' . $user['last_name'] : 'Unknown User';
    } catch (PDOException $e) {
        error_log("Get user name error: " . $e->getMessage());
        return 'Unknown User';
    }
}
?>