<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
    if (isCompany()) {
        redirect('company_dashboard.php');
    } elseif (isStore()) {
        redirect('store_dashboard.php');
    } elseif (isAdmin()) {
        redirect('admin_dashboard.php');
    }
}

// === ADD SECURITY HEADERS HERE ===
header('X-Frame-Options: DENY');
header('X-Content-Type-Options: nosniff');
header('X-XSS-Protection: 1; mode=block');

$errors = [];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // === ADD CSRF PROTECTION AND RATE LIMITING HERE ===
    if (!isset($_POST['csrf_token']) || !validateCSRFToken($_POST['csrf_token'])) {
        $errors[] = "Invalid security token. Please refresh the page and try again.";
    }
    
    // Rate limiting
    if (!rateLimit('login', 5, 900)) {
        $errors[] = "Too many login attempts. Please try again in 15 minutes.";
    }

    // Only proceed with login if no security errors
    if (empty($errors)) {
        $login_identifier = sanitize($_POST['login_identifier']);
        $password = $_POST['password'];
        $remember_me = isset($_POST['remember_me']);

        // Validation
        if (empty($login_identifier)) {
            $errors[] = "Email or phone number is required";
        }
        
        if (empty($password)) {
            $errors[] = "Password is required";
        }

        // Authentication
        if (empty($errors)) {
            // Check if identifier is email or phone
            $is_email = filter_var($login_identifier, FILTER_VALIDATE_EMAIL);
            
            if ($is_email) {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE email = ?");
            } else {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE phone = ?");
            }
            
            $stmt->execute([$login_identifier]);
            $user = $stmt->fetch(PDO::FETCH_ASSOC);

            if ($user && password_verify($password, $user['password'])) {
                // Record successful login attempt
                recordLoginAttempt($user['id'], true);
                
                // Set session variables
                $_SESSION['user_id'] = $user['id'];
                $_SESSION['first_name'] = $user['first_name'];
                $_SESSION['last_name'] = $user['last_name'];
                $_SESSION['email'] = $user['email'];
                $_SESSION['role'] = $user['role'];
                $_SESSION['state'] = $user['state'];
                $_SESSION['lga'] = $user['lga'];

                // Set remember me cookie if requested
                if ($remember_me) {
                    $token = bin2hex(random_bytes(32));
                    $expiry = time() + (30 * 24 * 60 * 60); // 30 days
                    
                    setcookie('remember_token', $token, $expiry, '/');
                }

                // Redirect based on role
                switch ($user['role']) {
                    case 'Company':
                        redirect('company_dashboard.php');
                        break;
                    case 'Provision Store':
                        redirect('store_dashboard.php');
                        break;
                    case 'Admin':
                        redirect('admin_dashboard.php');
                        break;
                    default:
                        $errors[] = "Unknown user role";
                        break;
                }
            } else {
                // Record failed login attempt
                if ($user) {
                    recordLoginAttempt($user['id'], false);
                }
                $errors[] = "Invalid email/phone or password";
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Login - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .login-container {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 50px 0;
            display: flex;
            align-items: center;
        }
        .login-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .login-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .login-body {
            padding: 40px;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .btn-login {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-weight: 600;
            width: 100%;
            transition: all 0.3s;
        }
        .btn-login:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        .register-link {
            color: #667eea;
            text-decoration: none;
        }
        .register-link:hover {
            text-decoration: underline;
        }
        .demo-accounts {
            background: #f8f9fa;
            border-radius: 10px;
            padding: 20px;
            margin-top: 20px;
        }
        .demo-account {
            padding: 10px;
            border-bottom: 1px solid #dee2e6;
        }
        .demo-account:last-child {
            border-bottom: none;
        }
        .input-group-text {
            background-color: #f8f9fa;
            border-right: none;
        }
        .form-control {
            border-left: none;
        }
        .password-toggle {
            cursor: pointer;
            background: #f8f9fa;
            border: 1px solid #ced4da;
            border-left: none;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-6 col-lg-5">
                    <div class="login-card">
                        <div class="login-header">
                            <h2><i class="fas fa-sign-in-alt me-2"></i>Welcome Back</h2>
                            <p class="mb-0">Sign in to your account</p>
                        </div>
                        
                        <div class="login-body">
                            <?php if (!empty($errors)): ?>
                                <div class="alert alert-danger">
                                    <ul class="mb-0">
                                        <?php foreach ($errors as $error): ?>
                                            <li><?php echo $error; ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>

                            <form method="POST" action="" id="loginForm">
                                <!-- === ADD CSRF TOKEN FIELD HERE === -->
                                <input type="hidden" name="csrf_token" value="<?php echo generateCSRFToken(); ?>">

                                <div class="mb-3">
                                    <label for="login_identifier" class="form-label">Email or Phone Number *</label>
                                    <div class="input-group">
                                        <span class="input-group-text">
                                            <i class="fas fa-user"></i>
                                        </span>
                                        <input type="text" class="form-control" id="login_identifier" name="login_identifier" 
                                               value="<?php echo isset($login_identifier) ? $login_identifier : ''; ?>" 
                                               placeholder="Enter your email or phone number" required>
                                    </div>
                                </div>

                                <div class="mb-3">
                                    <label for="password" class="form-label">Password *</label>
                                    <div class="input-group">
                                        <span class="input-group-text">
                                            <i class="fas fa-lock"></i>
                                        </span>
                                        <input type="password" class="form-control" id="password" name="password" 
                                               placeholder="Enter your password" required>
                                        <span class="input-group-text password-toggle" onclick="togglePassword()">
                                            <i class="fas fa-eye" id="passwordIcon"></i>
                                        </span>
                                    </div>
                                </div>

                                <div class="mb-3 d-flex justify-content-between align-items-center">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="remember_me" name="remember_me">
                                        <label class="form-check-label" for="remember_me">
                                            Remember me
                                        </label>
                                    </div>
                                    <a href="forgot_password.php" class="register-link">Forgot Password?</a>
                                </div>

                                <button type="submit" class="btn btn-login mb-3">
                                    <i class="fas fa-sign-in-alt me-2"></i>Sign In
                                </button>

                                <div class="text-center mb-3">
                                    <p>Don't have an account? <a href="signup.php" class="register-link">Register here</a></p>
                                </div>

                                <!-- Demo Accounts -->
                                <div class="demo-accounts">
                                    <h6 class="text-center mb-3">
                                        <i class="fas fa-info-circle me-2"></i>Demo Accounts
                                    </h6>
                                    
                                    <div class="demo-account">
                                        <small class="text-muted">Admin Account</small><br>
                                        <strong>Email:</strong> admin@multivendor.com<br>
                                        <strong>Password:</strong> password
                                    </div>
                                    
                                    <div class="demo-account">
                                        <small class="text-muted">Company Account</small><br>
                                        <strong>Email:</strong> company@example.com<br>
                                        <strong>Password:</strong> password
                                    </div>
                                    
                                    <div class="demo-account">
                                        <small class="text-muted">Store Account</small><br>
                                        <strong>Email:</strong> store@example.com<br>
                                        <strong>Phone:</strong> 08034567890<br>
                                        <strong>Password:</strong> password
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // === ADD SECURITY JAVASCRIPT HERE ===
        // Clear form on page load to prevent auto-fill
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('loginForm').reset();
        });

        // Prevent form resubmission on refresh
        if (window.history.replaceState) {
            window.history.replaceState(null, null, window.location.href);
        }

        // Password visibility toggle
        function togglePassword() {
            const passwordInput = document.getElementById('password');
            const passwordIcon = document.getElementById('passwordIcon');
            
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                passwordIcon.classList.remove('fa-eye');
                passwordIcon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                passwordIcon.classList.remove('fa-eye-slash');
                passwordIcon.classList.add('fa-eye');
            }
        }

        // Auto-detect if input is email or phone and format accordingly
        document.getElementById('login_identifier').addEventListener('input', function(e) {
            const value = e.target.value;
            
            // If it looks like a phone number (digits only), format it
            if (/^\d+$/.test(value.replace(/\s/g, ''))) {
                // Remove any existing formatting
                const cleanValue = value.replace(/\s/g, '');
                
                // Format as Nigerian phone number: 080 123 45678
                if (cleanValue.length <= 3) {
                    e.target.value = cleanValue;
                } else if (cleanValue.length <= 6) {
                    e.target.value = cleanValue.slice(0, 3) + ' ' + cleanValue.slice(3);
                } else {
                    e.target.value = cleanValue.slice(0, 3) + ' ' + cleanValue.slice(3, 6) + ' ' + cleanValue.slice(6, 11);
                }
            }
        });

        // Form validation
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            const identifier = document.getElementById('login_identifier').value;
            const password = document.getElementById('password').value;
            
            if (!identifier.trim()) {
                e.preventDefault();
                alert('Please enter your email or phone number!');
                return false;
            }
            
            if (!password.trim()) {
                e.preventDefault();
                alert('Please enter your password!');
                return false;
            }
        });

        // Quick fill demo accounts
        document.addEventListener('DOMContentLoaded', function() {
            // Add click handlers to demo accounts for quick filling
            const demoAccounts = document.querySelectorAll('.demo-account');
            
            demoAccounts.forEach(account => {
                account.style.cursor = 'pointer';
                account.addEventListener('click', function() {
                    const text = this.textContent;
                    
                    if (text.includes('admin@multivendor.com')) {
                        document.getElementById('login_identifier').value = 'admin@multivendor.com';
                        document.getElementById('password').value = 'password';
                    } else if (text.includes('company@example.com')) {
                        document.getElementById('login_identifier').value = 'company@example.com';
                        document.getElementById('password').value = 'password';
                    } else if (text.includes('store@example.com')) {
                        document.getElementById('login_identifier').value = 'store@example.com';
                        document.getElementById('password').value = 'password';
                    }
                });
            });
        });

        // Enter key submission
        document.addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                const form = document.getElementById('loginForm');
                if (form) {
                    form.dispatchEvent(new Event('submit'));
                }
            }
        });
    </script>
</body>
</html>