<?php
include 'includes/db_connect.php';
include 'includes/functions.php';
include 'includes/payment_config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

$gateway = $_GET['gateway'] ?? '';
$tx_ref = $_GET['tx_ref'] ?? '';
$transaction_id = $_GET['transaction_id'] ?? '';

if ($gateway === 'flutterwave') {
    verifyFlutterwavePayment($tx_ref);
} elseif ($gateway === 'aspay') {
    verifyAspayPayment($tx_ref);
} else {
    die("Invalid payment gateway");
}

function verifyFlutterwavePayment($tx_ref) {
    global $pdo;
    
    // Get transaction details
    $transaction_stmt = $pdo->prepare("
        SELECT t.*, o.store_id, o.company_id, o.product_id, o.quantity 
        FROM transactions t 
        JOIN orders o ON t.order_id = o.id 
        WHERE t.transaction_ref = ?
    ");
    $transaction_stmt->execute([$tx_ref]);
    $transaction = $transaction_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$transaction) {
        redirect('payment_failed.php?reason=transaction_not_found');
    }
    
    // Verify with Flutterwave API
    $curl = curl_init();
    curl_setopt_array($curl, [
        CURLOPT_URL => PaymentConfig::$flutterwave_verify_url . $tx_ref . '/verify',
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_ENCODING => '',
        CURLOPT_MAXREDIRS => 10,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
        CURLOPT_CUSTOMREQUEST => 'GET',
        CURLOPT_HTTPHEADER => [
            'Authorization: Bearer ' . PaymentConfig::$flutterwave_secret_key,
            'Content-Type: application/json'
        ],
    ]);
    
    $response = curl_exec($curl);
    $err = curl_error($curl);
    curl_close($curl);
    
    if ($err) {
        // Update transaction as failed
        updateTransactionStatus($transaction['id'], 'Failed', $tx_ref);
        redirect('payment_failed.php?reason=verification_error');
    }
    
    $response_data = json_decode($response, true);
    
    if ($response_data['status'] === 'success' && $response_data['data']['status'] === 'successful') {
        // Payment successful
        $amount_paid = $response_data['data']['amount'];
        $currency = $response_data['data']['currency'];
        
        if ($amount_paid >= $transaction['amount'] && $currency === PaymentConfig::$currency) {
            // Update transaction as completed
            updateTransactionStatus($transaction['id'], 'Completed', $tx_ref);
            
            // Update order payment status
            updateOrderPaymentStatus($transaction['order_id'], 'Paid', $tx_ref);
            
            redirect('payment_success.php?order_id=' . $transaction['order_id'] . '&tx_ref=' . $tx_ref);
        } else {
            updateTransactionStatus($transaction['id'], 'Failed', $tx_ref);
            redirect('payment_failed.php?reason=amount_mismatch');
        }
    } else {
        updateTransactionStatus($transaction['id'], 'Failed', $tx_ref);
        redirect('payment_failed.php?reason=payment_failed');
    }
}

function verifyAspayPayment($tx_ref) {
    // Similar implementation for Aspay
    // This would contain Aspay-specific verification logic
    global $pdo;
    
    // Placeholder for Aspay verification
    // In production, implement actual Aspay API integration
    
    $transaction_stmt = $pdo->prepare("SELECT * FROM transactions WHERE transaction_ref = ?");
    $transaction_stmt->execute([$tx_ref]);
    $transaction = $transaction_stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($transaction) {
        // Simulate successful payment for demo
        updateTransactionStatus($transaction['id'], 'Completed', $tx_ref);
        updateOrderPaymentStatus($transaction['order_id'], 'Paid', $tx_ref);
        redirect('payment_success.php?order_id=' . $transaction['order_id'] . '&tx_ref=' . $tx_ref);
    } else {
        redirect('payment_failed.php?reason=transaction_not_found');
    }
}

function updateTransactionStatus($transaction_id, $status, $tx_ref) {
    global $pdo;
    $stmt = $pdo->prepare("UPDATE transactions SET status = ?, transaction_ref = ? WHERE id = ?");
    $stmt->execute([$status, $tx_ref, $transaction_id]);
}

function updateOrderPaymentStatus($order_id, $status, $payment_ref) {
    global $pdo;
    $stmt = $pdo->prepare("UPDATE orders SET payment_status = ?, payment_reference = ? WHERE id = ?");
    $stmt->execute([$status, $payment_ref, $order_id]);
}

redirect('payment_failed.php?reason=unknown_error');
?>