<?php
include 'includes/db_connect.php';
include 'includes/functions.php';
include 'includes/payment_config.php';

if (!isLoggedIn()) {
    redirect('login.php');
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $order_id = intval($_POST['order_id']);
    $amount = floatval($_POST['amount']);
    $user_id = $_SESSION['user_id'];
    
    // Verify order belongs to user
    $order_stmt = $pdo->prepare("
        SELECT o.*, u.email, u.phone, u.first_name, u.last_name 
        FROM orders o 
        JOIN users u ON o.store_id = u.id 
        WHERE o.id = ? AND o.store_id = ?
    ");
    $order_stmt->execute([$order_id, $user_id]);
    $order = $order_stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$order) {
        die("Order not found");
    }
    
    // Generate transaction reference
    $tx_ref = generateTransactionRef();
    
    // Create transaction record
    $transaction_stmt = $pdo->prepare("
        INSERT INTO transactions (order_id, amount, payment_method, payment_gateway, status, transaction_ref, customer_email, customer_phone) 
        VALUES (?, ?, 'Card', 'Flutterwave', 'Pending', ?, ?, ?)
    ");
    $transaction_stmt->execute([$order_id, $amount, $tx_ref, $order['email'], $order['phone']]);
    
    $transaction_id = $pdo->lastInsertId();
    
    // Prepare payment data for Flutterwave
    $payment_data = [
        'tx_ref' => $tx_ref,
        'amount' => $amount,
        'currency' => PaymentConfig::$currency,
        'redirect_url' => 'http://' . $_SERVER['HTTP_HOST'] . '/B2Store/payment_verify.php?gateway=flutterwave&tx_ref=' . $tx_ref,
        'customer' => [
            'email' => $order['email'],
            'phone_number' => $order['phone'],
            'name' => $order['first_name'] . ' ' . $order['last_name']
        ],
        'customizations' => [
            'title' => 'MultiVendor Order Payment',
            'description' => 'Payment for Order #' . $order_id,
            'logo' => 'http://' . $_SERVER['HTTP_HOST'] . '/B2Store/assets/images/logo.png'
        ],
        'meta' => [
            'order_id' => $order_id,
            'user_id' => $user_id,
            'transaction_id' => $transaction_id
        ]
    ];
    
    // Store payment data in session for verification
    $_SESSION['payment_data'] = $payment_data;
    $_SESSION['current_transaction'] = $transaction_id;
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Processing Payment - MultiVendor System</title>
    <script src="https://checkout.flutterwave.com/v3.js"></script>
    <style>
        .payment-container {
            display: flex;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        }
        .payment-card {
            background: white;
            border-radius: 15px;
            padding: 40px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            text-align: center;
            max-width: 500px;
            width: 90%;
        }
        .loading-spinner {
            border: 4px solid #f3f3f3;
            border-top: 4px solid #667eea;
            border-radius: 50%;
            width: 50px;
            height: 50px;
            animation: spin 1s linear infinite;
            margin: 0 auto 20px;
        }
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body>
    <div class="payment-container">
        <div class="payment-card">
            <div class="loading-spinner"></div>
            <h3>Processing Your Payment</h3>
            <p>You will be redirected to Flutterwave to complete your payment.</p>
            <p class="text-muted">Do not close this window.</p>
            
            <button id="payButton" class="btn btn-primary btn-lg mt-3" style="display: none;">
                Proceed to Payment
            </button>
        </div>
    </div>

    <script>
        const paymentData = <?php echo json_encode($payment_data); ?>;
        
        function makePayment() {
            FlutterwaveCheckout({
                public_key: "<?php echo PaymentConfig::$flutterwave_public_key; ?>",
                tx_ref: paymentData.tx_ref,
                amount: paymentData.amount,
                currency: paymentData.currency,
                payment_options: "card, banktransfer, ussd",
                redirect_url: paymentData.redirect_url,
                customer: paymentData.customer,
                customizations: paymentData.customizations,
                meta: paymentData.meta,
                onclose: function() {
                    alert("Payment window closed.");
                },
            });
        }
        
        // Auto-trigger payment after short delay
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(() => {
                makePayment();
            }, 2000);
        });
    </script>
</body>
</html>