<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

// Redirect if already logged in
if (isLoggedIn()) {
    if (isCompany()) {
        redirect('company_dashboard.php');
    } elseif (isStore()) {
        redirect('store_dashboard.php');
    } elseif (isAdmin()) {
        redirect('admin_dashboard.php');
    }
}

$errors = [];
$success = '';

// Nigerian States and LGAs data
$nigerian_states = [
    'Abia' => ['Aba North', 'Aba South', 'Arochukwu', 'Bende', 'Ikawuno', 'Ikwuano', 'Isiala Ngwa North', 'Isiala Ngwa South', 'Isuikwuato', 'Umu Nneochi', 'Obi Ngwa', 'Osisoma Ngwa', 'Ugwunagbo', 'Ukwa East', 'Ukwa West', 'Umuahia North', 'Umuahia South'],
    'Adamawa' => ['Demsa', 'Fufore', 'Ganye', 'Girei', 'Gombi', 'Guyuk', 'Hong', 'Jada', 'Lamurde', 'Madagali', 'Maiha', 'Mayo Belwa', 'Michika', 'Mubi North', 'Mubi South', 'Numan', 'Shelleng', 'Song', 'Toungo', 'Yola North', 'Yola South'],
    'Lagos' => ['Agege', 'Ajeromi-Ifelodun', 'Alimosho', 'Amuwo-Odofin', 'Apapa', 'Badagry', 'Epe', 'Eti Osa', 'Ibeju-Lekki', 'Ifako-Ijaiye', 'Ikeja', 'Ikorodu', 'Kosofe', 'Lagos Island', 'Lagos Mainland', 'Mushin', 'Ojo', 'Oshodi-Isolo', 'Shomolu', 'Surulere'],
    'Rivers' => ['Port Harcourt', 'Obio-Akpor', 'Okrika', 'Ogu–Bolo', 'Eleme', 'Tai', 'Gokana', 'Khana', 'Oyigbo', 'Opobo–Nkoro', 'Andoni', 'Bonny', 'Degema', 'Asari-Toru', 'Akuku-Toru', 'Abua–Odual', 'Ahoada West', 'Ahoada East', 'Ogba–Egbema–Ndoni', 'Emohua', 'Ikwerre', 'Etche', 'Omuma']
];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Sanitize inputs
    $first_name = sanitize($_POST['first_name']);
    $last_name = sanitize($_POST['last_name']);
    $other_name = sanitize($_POST['other_name']);
    $email = sanitize($_POST['email']);
    $phone = sanitize($_POST['phone']);
    $role = sanitize($_POST['role']);
    $state = sanitize($_POST['state']);
    $lga = sanitize($_POST['lga']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];

    // Validation
    if (empty($first_name)) {
        $errors[] = "First name is required";
    }
    
    if (empty($last_name)) {
        $errors[] = "Last name is required";
    }
    
    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Valid email is required";
    }
    
    if (empty($phone) || strlen($phone) < 10) {
        $errors[] = "Valid phone number is required";
    }
    
    if (empty($role) || !in_array($role, ['Company', 'Provision Store'])) {
        $errors[] = "Please select a valid role";
    }
    
    if (empty($state) || !array_key_exists($state, $nigerian_states)) {
        $errors[] = "Please select a valid state";
    }
    
    if (empty($lga) || !in_array($lga, $nigerian_states[$state])) {
        $errors[] = "Please select a valid LGA";
    }
    
    if (strlen($password) < 6) {
        $errors[] = "Password must be at least 6 characters long";
    }
    
    if ($password !== $confirm_password) {
        $errors[] = "Passwords do not match";
    }

    // Check if email or phone already exists
    if (empty($errors)) {
        $check_stmt = $pdo->prepare("SELECT id FROM users WHERE email = ? OR phone = ?");
        $check_stmt->execute([$email, $phone]);
        
        if ($check_stmt->rowCount() > 0) {
            $errors[] = "Email or phone number already registered";
        }
    }

    // Insert user if no errors
    if (empty($errors)) {
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        $insert_stmt = $pdo->prepare("INSERT INTO users (first_name, last_name, other_name, email, phone, role, state, lga, password) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
        
        if ($insert_stmt->execute([$first_name, $last_name, $other_name, $email, $phone, $role, $state, $lga, $hashed_password])) {
            $success = "Registration successful! You can now login.";
            // Clear form
            $first_name = $last_name = $other_name = $email = $phone = $role = $state = $lga = '';
        } else {
            $errors[] = "Registration failed. Please try again.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .register-container {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 50px 0;
        }
        .register-card {
            background: white;
            border-radius: 15px;
            box-shadow: 0 15px 35px rgba(0,0,0,0.1);
            overflow: hidden;
        }
        .register-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 30px;
            text-align: center;
        }
        .register-body {
            padding: 40px;
        }
        .form-control:focus {
            border-color: #667eea;
            box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.25);
        }
        .btn-register {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            padding: 12px 30px;
            border-radius: 25px;
            font-weight: 600;
            width: 100%;
            transition: all 0.3s;
        }
        .btn-register:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.2);
        }
        .login-link {
            color: #667eea;
            text-decoration: none;
        }
        .login-link:hover {
            text-decoration: underline;
        }
        .role-option {
            border: 2px solid #e9ecef;
            border-radius: 10px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
            margin-bottom: 10px;
        }
        .role-option:hover {
            border-color: #667eea;
            background-color: #f8f9ff;
        }
        .role-option.selected {
            border-color: #667eea;
            background-color: #f0f2ff;
        }
        .role-icon {
            font-size: 2rem;
            margin-bottom: 10px;
            color: #667eea;
        }
    </style>
</head>
<body>
    <div class="register-container">
        <div class="container">
            <div class="row justify-content-center">
                <div class="col-md-8 col-lg-6">
                    <div class="register-card">
                        <div class="register-header">
                            <h2><i class="fas fa-user-plus me-2"></i>Create Account</h2>
                            <p class="mb-0">Join our multi-vendor platform today</p>
                        </div>
                        
                        <div class="register-body">
                            <?php if (!empty($errors)): ?>
                                <div class="alert alert-danger">
                                    <ul class="mb-0">
                                        <?php foreach ($errors as $error): ?>
                                            <li><?php echo $error; ?></li>
                                        <?php endforeach; ?>
                                    </ul>
                                </div>
                            <?php endif; ?>
                            
                            <?php if ($success): ?>
                                <div class="alert alert-success">
                                    <?php echo $success; ?>
                                    <br>
                                    <a href="login.php" class="alert-link">Click here to login</a>
                                </div>
                            <?php endif; ?>

                            <form method="POST" action="" id="registrationForm">
                                <div class="row">
                                    <div class="col-md-4 mb-3">
                                        <label for="first_name" class="form-label">First Name *</label>
                                        <input type="text" class="form-control" id="first_name" name="first_name" 
                                               value="<?php echo isset($first_name) ? $first_name : ''; ?>" required>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="last_name" class="form-label">Last Name *</label>
                                        <input type="text" class="form-control" id="last_name" name="last_name" 
                                               value="<?php echo isset($last_name) ? $last_name : ''; ?>" required>
                                    </div>
                                    <div class="col-md-4 mb-3">
                                        <label for="other_name" class="form-label">Other Name</label>
                                        <input type="text" class="form-control" id="other_name" name="other_name" 
                                               value="<?php echo isset($other_name) ? $other_name : ''; ?>">
                                    </div>
                                </div>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="email" class="form-label">Email Address *</label>
                                        <input type="email" class="form-control" id="email" name="email" 
                                               value="<?php echo isset($email) ? $email : ''; ?>" required>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="phone" class="form-label">Phone Number *</label>
                                        <input type="tel" class="form-control" id="phone" name="phone" 
                                               value="<?php echo isset($phone) ? $phone : ''; ?>" required>
                                    </div>
                                </div>

                                <!-- Role Selection -->
                                <div class="mb-4">
                                    <label class="form-label">Account Type *</label>
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="role-option" onclick="selectRole('Company')">
                                                <div class="role-icon">
                                                    <i class="fas fa-industry"></i>
                                                </div>
                                                <h5>Company</h5>
                                                <p class="text-muted small">Sell products to provision stores</p>
                                                <input type="radio" name="role" value="Company" style="display: none;">
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="role-option" onclick="selectRole('Provision Store')">
                                                <div class="role-icon">
                                                    <i class="fas fa-store"></i>
                                                </div>
                                                <h5>Provision Store</h5>
                                                <p class="text-muted small">Buy products from companies</p>
                                                <input type="radio" name="role" value="Provision Store" style="display: none;">
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="state" class="form-label">State *</label>
                                        <select class="form-select" id="state" name="state" required>
                                            <option value="">Select State</option>
                                            <?php foreach ($nigerian_states as $state_name => $lgas): ?>
                                                <option value="<?php echo $state_name; ?>" 
                                                    <?php echo (isset($state) && $state === $state_name) ? 'selected' : ''; ?>>
                                                    <?php echo $state_name; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        </select>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="lga" class="form-label">LGA *</label>
                                        <select class="form-select" id="lga" name="lga" required>
                                            <option value="">Select LGA</option>
                                            <?php if (isset($state) && !empty($state)): ?>
                                                <?php foreach ($nigerian_states[$state] as $lga_name): ?>
                                                    <option value="<?php echo $lga_name; ?>"
                                                        <?php echo (isset($lga) && $lga === $lga_name) ? 'selected' : ''; ?>>
                                                        <?php echo $lga_name; ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </select>
                                    </div>
                                </div>

                                <div class="row">
                                    <div class="col-md-6 mb-3">
                                        <label for="password" class="form-label">Password *</label>
                                        <input type="password" class="form-control" id="password" name="password" required>
                                        <div class="form-text">Minimum 6 characters</div>
                                    </div>
                                    <div class="col-md-6 mb-3">
                                        <label for="confirm_password" class="form-label">Confirm Password *</label>
                                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                                    </div>
                                </div>

                                <div class="mb-4">
                                    <div class="form-check">
                                        <input class="form-check-input" type="checkbox" id="terms" required>
                                        <label class="form-check-label" for="terms">
                                            I agree to the <a href="#" class="login-link">Terms of Service</a> and <a href="#" class="login-link">Privacy Policy</a>
                                        </label>
                                    </div>
                                </div>

                                <button type="submit" class="btn btn-register mb-3">
                                    <i class="fas fa-user-plus me-2"></i>Create Account
                                </button>

                                <div class="text-center">
                                    <p>Already have an account? <a href="login.php" class="login-link">Login here</a></p>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Role selection
        function selectRole(role) {
            // Remove selected class from all options
            document.querySelectorAll('.role-option').forEach(option => {
                option.classList.remove('selected');
            });
            
            // Add selected class to clicked option
            event.currentTarget.classList.add('selected');
            
            // Set the radio button value
            const radio = event.currentTarget.querySelector('input[type="radio"]');
            radio.checked = true;
        }

        // Dynamic LGA loading based on state selection
        document.getElementById('state').addEventListener('change', function() {
            const state = this.value;
            const lgaSelect = document.getElementById('lga');
            
            // Nigerian States and LGAs data
            const statesLgas = <?php echo json_encode($nigerian_states); ?>;
            
            // Clear existing options
            lgaSelect.innerHTML = '<option value="">Select LGA</option>';
            
            if (state && statesLgas[state]) {
                statesLgas[state].forEach(lga => {
                    const option = document.createElement('option');
                    option.value = lga;
                    option.textContent = lga;
                    lgaSelect.appendChild(option);
                });
            }
        });

        // Password confirmation validation
        document.getElementById('registrationForm').addEventListener('submit', function(e) {
            const password = document.getElementById('password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            
            if (password !== confirmPassword) {
                e.preventDefault();
                alert('Passwords do not match!');
                return false;
            }
            
            if (password.length < 6) {
                e.preventDefault();
                alert('Password must be at least 6 characters long!');
                return false;
            }
            
            const roleSelected = document.querySelector('input[name="role"]:checked');
            if (!roleSelected) {
                e.preventDefault();
                alert('Please select an account type!');
                return false;
            }
        });

        // Initialize role selection if form was submitted with errors
        document.addEventListener('DOMContentLoaded', function() {
            const selectedRole = "<?php echo isset($role) ? $role : ''; ?>";
            if (selectedRole) {
                const roleOption = document.querySelector(`.role-option input[value="${selectedRole}"]`);
                if (roleOption) {
                    roleOption.parentElement.classList.add('selected');
                    roleOption.checked = true;
                }
            }
        });
    </script>
</body>
</html>