<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

// Check if user is logged in and is a provision store
if (!isLoggedIn() || !isStore()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$state = $_SESSION['state'];
$lga = $_SESSION['lga'];

// Get store statistics
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(orders.id) as total_orders,
        SUM(CASE WHEN orders.status = 'Delivered' THEN 1 ELSE 0 END) as delivered_orders,
        SUM(CASE WHEN orders.status = 'Pending' OR orders.status = 'Processing' THEN 1 ELSE 0 END) as pending_orders,
        SUM(CASE WHEN orders.payment_status = 'Paid' THEN orders.total_price ELSE 0 END) as total_spent
    FROM orders 
    WHERE store_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);

// Get recent orders - FIXED: Removed company_name and store_name references
$recent_orders_stmt = $pdo->prepare("
    SELECT o.*, p.name as product_name, u.first_name, u.last_name
    FROM orders o 
    JOIN products p ON o.product_id = p.id 
    JOIN users u ON o.company_id = u.id 
    WHERE o.store_id = ? 
    ORDER BY o.created_at DESC 
    LIMIT 5
");
$recent_orders_stmt->execute([$user_id]);
$recent_orders = $recent_orders_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get available companies in same state - FIXED: Removed company_name reference
$companies_stmt = $pdo->prepare("
    SELECT DISTINCT u.id, u.first_name, u.last_name
    FROM users u 
    JOIN products p ON u.id = p.company_id 
    WHERE u.role = 'Company' AND u.state = ? AND u.lga = ?
    LIMIT 5
");
$companies_stmt->execute([$state, $lga]);
$local_companies = $companies_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get product categories
$categories_stmt = $pdo->prepare("
    SELECT DISTINCT category 
    FROM products 
    JOIN users ON products.company_id = users.id 
    WHERE users.state = ? AND users.lga = ?
    LIMIT 10
");
$categories_stmt->execute([$state, $lga]);
$categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Store Dashboard - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .sidebar {
            background: linear-gradient(135deg, #42b883 0%, #47b8a4 100%);
            color: white;
            min-height: 100vh;
            position: fixed;
            width: 250px;
            transition: all 0.3s;
        }
        .sidebar .nav-link {
            color: white;
            padding: 15px 20px;
            border-bottom: 1px solid rgba(255,255,255,0.1);
            transition: all 0.3s;
        }
        .sidebar .nav-link:hover, .sidebar .nav-link.active {
            background: rgba(255,255,255,0.1);
            padding-left: 30px;
        }
        .sidebar .nav-link i {
            width: 20px;
            margin-right: 10px;
        }
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .navbar-custom {
            background: white;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .stat-card {
            border: none;
            border-radius: 15px;
            transition: transform 0.3s, box-shadow 0.3s;
            margin-bottom: 20px;
        }
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        .stat-icon {
            font-size: 2.5rem;
            opacity: 0.7;
        }
        .bg-primary { background: linear-gradient(135deg, #42b883, #47b8a4); }
        .bg-success { background: linear-gradient(135deg, #4facfe, #00f2fe); }
        .bg-warning { background: linear-gradient(135deg, #ff9a44, #ff6b6b); }
        .bg-info { background: linear-gradient(135deg, #667eea, #764ba2); }
        .company-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 15px;
        }
        .company-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .category-badge {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 20px;
            padding: 8px 15px;
            margin: 5px;
            cursor: pointer;
            transition: all 0.3s;
        }
        .category-badge:hover, .category-badge.active {
            background: #42b883;
            color: white;
            border-color: #42b883;
        }
        @media (max-width: 768px) {
            .sidebar {
                margin-left: -250px;
            }
            .sidebar.active {
                margin-left: 0;
            }
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar" id="sidebar">
        <div class="sidebar-header text-center py-4">
            <h4><i class="fas fa-store me-2"></i>Store Panel</h4>
            <p class="mb-0 text-light"><?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?></p>
            <small class="text-light"><?php echo $state . ', ' . $lga; ?></small>
        </div>
        <nav class="nav flex-column">
            <a class="nav-link active" href="store_dashboard.php">
                <i class="fas fa-tachometer-alt"></i>Dashboard
            </a>
            <a class="nav-link" href="store_orders.php">
                <i class="fas fa-shopping-cart"></i>Make Order
            </a>
            <a class="nav-link" href="store_order_history.php">
                <i class="fas fa-history"></i>Order History
            </a>
            <a class="nav-link" href="store_transactions.php">
                <i class="fas fa-money-bill-wave"></i>Transactions
            </a>
            <a class="nav-link" href="store_messages.php">
                <i class="fas fa-comments"></i>Support
            </a>
            <a class="nav-link" href="store_profile.php">
                <i class="fas fa-user-cog"></i>Profile
            </a>
            <a class="nav-link" href="logout.php">
                <i class="fas fa-sign-out-alt"></i>Logout
            </a>
        </nav>
    </div>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-custom mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-success me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">Store Dashboard</span>
                <div class="navbar-nav ms-auto">
                    <div class="nav-item dropdown">
                        <a class="nav-link dropdown-toggle" href="#" id="navbarDropdown" role="button" data-bs-toggle="dropdown">
                            <i class="fas fa-store me-1"></i>
                            <?php echo $_SESSION['first_name'] . ' ' . $_SESSION['last_name']; ?>
                        </a>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="store_profile.php"><i class="fas fa-user me-2"></i>Profile</a></li>
                            <li><hr class="dropdown-divider"></li>
                            <li><a class="dropdown-item" href="logout.php"><i class="fas fa-sign-out-alt me-2"></i>Logout</a></li>
                        </ul>
                    </div>
                </div>
            </div>
        </nav>

        <!-- Statistics Cards -->
        <div class="row">
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-primary stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['total_orders'] ?? 0; ?></h4>
                                <p class="mb-0">Total Orders</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-shopping-cart"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-success stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['delivered_orders'] ?? 0; ?></h4>
                                <p class="mb-0">Delivered</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-check-circle"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-warning stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0"><?php echo $stats['pending_orders'] ?? 0; ?></h4>
                                <p class="mb-0">Pending</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-clock"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <div class="col-xl-3 col-md-6">
                <div class="card text-white bg-info stat-card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <div>
                                <h4 class="mb-0">₦<?php echo number_format($stats['total_spent'] ?? 0, 2); ?></h4>
                                <p class="mb-0">Total Spent</p>
                            </div>
                            <div class="stat-icon">
                                <i class="fas fa-money-bill-wave"></i>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row mt-4">
            <!-- Recent Orders -->
            <div class="col-lg-7">
                <div class="card">
                    <div class="card-header d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-clock me-2"></i>Recent Orders
                        </h5>
                        <a href="store_order_history.php" class="btn btn-sm btn-success">View All</a>
                    </div>
                    <div class="card-body">
                        <?php if ($recent_orders): ?>
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>Product</th>
                                            <th>Company</th>
                                            <th>Qty</th>
                                            <th>Amount</th>
                                            <th>Status</th>
                                            <th>Date</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php foreach ($recent_orders as $order): ?>
                                            <tr>
                                                <td><?php echo $order['product_name']; ?></td>
                                                <td><?php echo $order['first_name'] . ' ' . $order['last_name']; ?></td>
                                                <td><?php echo $order['quantity']; ?></td>
                                                <td>₦<?php echo number_format($order['total_price'], 2); ?></td>
                                                <td>
                                                    <span class="badge 
                                                        <?php 
                                                        switch($order['status']) {
                                                            case 'Delivered': echo 'bg-success'; break;
                                                            case 'Processing': echo 'bg-primary'; break;
                                                            case 'Pending': echo 'bg-warning'; break;
                                                            case 'Cancelled': echo 'bg-danger'; break;
                                                            default: echo 'bg-secondary';
                                                        }
                                                        ?>">
                                                        <?php echo $order['status']; ?>
                                                    </span>
                                                    <br>
                                                    <small class="text-muted">
                                                        Payment: 
                                                        <span class="badge 
                                                            <?php 
                                                            switch($order['payment_status']) {
                                                                case 'Paid': echo 'bg-success'; break;
                                                                case 'Pending': echo 'bg-warning'; break;
                                                                case 'Failed': echo 'bg-danger'; break;
                                                                default: echo 'bg-secondary';
                                                            }
                                                            ?>">
                                                            <?php echo $order['payment_status']; ?>
                                                        </span>
                                                    </small>
                                                </td>
                                                <td><?php echo date('M j, Y', strtotime($order['created_at'])); ?></td>
                                            </tr>
                                        <?php endforeach; ?>
                                    </tbody>
                                </table>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-4">
                                <i class="fas fa-shopping-cart fa-3x text-muted mb-3"></i>
                                <p class="text-muted">No orders yet</p>
                                <a href="store_orders.php" class="btn btn-success">Make Your First Order</a>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

            <!-- Quick Access & Local Companies -->
            <div class="col-lg-5">
                <!-- Quick Order -->
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-bolt me-2"></i>Quick Order
                        </h5>
                    </div>
                    <div class="card-body">
                        <div class="mb-3">
                            <label class="form-label">Browse by Category</label>
                            <div class="d-flex flex-wrap">
                                <?php foreach ($categories as $category): ?>
                                    <span class="category-badge" onclick="filterByCategory('<?php echo $category['category']; ?>')">
                                        <?php echo $category['category']; ?>
                                    </span>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        <div class="d-grid gap-2">
                            <a href="store_orders.php" class="btn btn-success">
                                <i class="fas fa-search me-2"></i>Browse Products
                            </a>
                            <a href="store_orders.php?new=true" class="btn btn-outline-success">
                                <i class="fas fa-plus me-2"></i>New Order
                            </a>
                        </div>
                    </div>
                </div>

                <!-- Local Companies -->
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-building me-2"></i>Local Companies
                            <small class="text-muted">(<?php echo $state . ', ' . $lga; ?>)</small>
                        </h5>
                    </div>
                    <div class="card-body">
                        <?php if ($local_companies): ?>
                            <div class="list-group list-group-flush">
                                <?php foreach ($local_companies as $company): ?>
                                    <div class="list-group-item d-flex justify-content-between align-items-center">
                                        <div>
                                            <h6 class="mb-1"><?php echo $company['first_name'] . ' ' . $company['last_name']; ?></h6>
                                            <small class="text-muted">Available in your area</small>
                                        </div>
                                        <div>
                                            <a href="store_messages.php?company=<?php echo $company['id']; ?>" class="btn btn-sm btn-outline-success" title="Message">
                                                <i class="fas fa-comment"></i>
                                            </a>
                                            <a href="store_orders.php?company=<?php echo $company['id']; ?>" class="btn btn-sm btn-success" title="View Products">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                            <div class="text-center mt-3">
                                <a href="store_orders.php" class="btn btn-sm btn-outline-success">View All Companies</a>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-3">
                                <i class="fas fa-building fa-2x text-muted mb-2"></i>
                                <p class="text-muted mb-0">No companies in your area yet</p>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Sidebar toggle for mobile
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            document.getElementById('sidebar').classList.toggle('active');
        });

        // Category filter function
        function filterByCategory(category) {
            window.location.href = `store_orders.php?category=${encodeURIComponent(category)}`;
        }

        // Auto-hide sidebar on mobile when clicking outside
        document.addEventListener('click', function(event) {
            const sidebar = document.getElementById('sidebar');
            const sidebarToggle = document.getElementById('sidebarToggle');
            
            if (window.innerWidth <= 768 && 
                !sidebar.contains(event.target) && 
                !sidebarToggle.contains(event.target) &&
                sidebar.classList.contains('active')) {
                sidebar.classList.remove('active');
            }
        });

        // Handle window resize
        window.addEventListener('resize', function() {
            if (window.innerWidth > 768) {
                document.getElementById('sidebar').classList.remove('active');
            }
        });

        // Real-time order status updates
        function checkOrderUpdates() {
            // This would typically make an AJAX call to check for order status updates
            console.log('Checking for order updates...');
        }

        // Check for updates every 60 seconds
        setInterval(checkOrderUpdates, 60000);
    </script>
</body>
</html>