<?php
include 'includes/db_connect.php';
include 'includes/functions.php';
include 'includes/message_functions.php';

if (!isLoggedIn() || !isStore()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$state = $_SESSION['state'];
$lga = $_SESSION['lga'];

// Handle sending new message
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['send_message'])) {
    $receiver_id = intval($_POST['receiver_id']);
    $subject = sanitize($_POST['subject']);
    $message = sanitize($_POST['message']);
    $message_type = $_POST['message_type'] ?? 'General';
    
    if (sendMessage($user_id, $receiver_id, $message, $subject, $message_type)) {
        $_SESSION['success'] = "Message sent successfully!";
    } else {
        $_SESSION['error'] = "Failed to send message. Please try again.";
    }
}

// Get conversation with specific user
$conversation_user_id = $_GET['user_id'] ?? null;
$conversation = [];
if ($conversation_user_id) {
    $conversation = getConversation($user_id, $conversation_user_id);
    markMessagesAsRead($user_id, $conversation_user_id);
}

// Get all message threads
$threads = getMessageThreads($user_id);

// Get companies for new message
$companies_stmt = $pdo->prepare("
    SELECT id, first_name, last_name, company_name 
    FROM users 
    WHERE role = 'Company' AND state = ? AND lga = ?
    ORDER BY first_name, last_name
");
$companies_stmt->execute([$state, $lga]);
$local_companies = $companies_stmt->fetchAll(PDO::FETCH_ASSOC);

$unread_count = getUnreadMessageCount($user_id);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Messages - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .messages-container {
            background: #f8f9fa;
            border-radius: 10px;
            height: 70vh;
            overflow: hidden;
        }
        .threads-list {
            border-right: 1px solid #dee2e6;
            height: 100%;
            overflow-y: auto;
        }
        .thread-item {
            border-bottom: 1px solid #dee2e6;
            padding: 15px;
            cursor: pointer;
            transition: background-color 0.3s;
        }
        .thread-item:hover, .thread-item.active {
            background-color: #e9ecef;
        }
        .thread-item.unread {
            background-color: #e7f3ff;
            font-weight: 500;
        }
        .messages-area {
            height: 100%;
            display: flex;
            flex-direction: column;
        }
        .messages-list {
            flex-grow: 1;
            overflow-y: auto;
            padding: 20px;
            background: white;
        }
        .message {
            margin-bottom: 15px;
            padding: 10px 15px;
            border-radius: 10px;
            max-width: 70%;
        }
        .message.sent {
            background: #007bff;
            color: white;
            margin-left: auto;
        }
        .message.received {
            background: #e9ecef;
            color: #333;
        }
        .message-form {
            border-top: 1px solid #dee2e6;
            padding: 15px;
            background: white;
        }
        .unread-badge {
            background: #dc3545;
        }
        .chat-header {
            background: white;
            border-bottom: 1px solid #dee2e6;
            padding: 15px;
        }
    </style>
</head>
<body>
    <?php include 'includes/store_sidebar.php'; ?>

    <div class="main-content" style="margin-left: 250px; padding: 20px;">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-comments me-2"></i>Messages
                    <?php if ($unread_count > 0): ?>
                        <span class="badge bg-danger"><?php echo $unread_count; ?> unread</span>
                    <?php endif; ?>
                </span>
                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#newMessageModal">
                    <i class="fas fa-plus me-2"></i>New Message
                </button>
            </div>
        </nav>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <div class="row">
            <div class="col-md-4">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title mb-0">Conversations</h5>
                    </div>
                    <div class="card-body p-0">
                        <div class="threads-list">
                            <?php if ($threads): ?>
                                <?php foreach ($threads as $thread): ?>
                                    <div class="thread-item <?php echo $thread['unread_count'] > 0 ? 'unread' : ''; ?> <?php echo $conversation_user_id == $thread['other_user_id'] ? 'active' : ''; ?>" 
                                         onclick="location.href='?user_id=<?php echo $thread['other_user_id']; ?>'">
                                        <div class="d-flex justify-content-between align-items-start">
                                            <div>
                                                <h6 class="mb-1">
                                                    <?php echo $thread['company_name'] ?? $thread['first_name'] . ' ' . $thread['last_name']; ?>
                                                </h6>
                                                <p class="mb-1 text-muted small">
                                                    <?php echo strlen($thread['message']) > 50 ? substr($thread['message'], 0, 50) . '...' : $thread['message']; ?>
                                                </p>
                                                <small class="text-muted">
                                                    <?php echo date('M j, g:i A', strtotime($thread['created_at'])); ?>
                                                </small>
                                            </div>
                                            <?php if ($thread['unread_count'] > 0): ?>
                                                <span class="badge unread-badge"><?php echo $thread['unread_count']; ?></span>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <div class="text-center py-5">
                                    <i class="fas fa-comments fa-2x text-muted mb-3"></i>
                                    <p class="text-muted">No messages yet</p>
                                    <button class="btn btn-primary btn-sm" data-bs-toggle="modal" data-bs-target="#newMessageModal">
                                        Start a conversation
                                    </button>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-8">
                <div class="card">
                    <div class="card-body p-0">
                        <?php if ($conversation_user_id): ?>
                            <?php 
                            $other_user = null;
                            foreach ($threads as $thread) {
                                if ($thread['other_user_id'] == $conversation_user_id) {
                                    $other_user = $thread;
                                    break;
                                }
                            }
                            ?>
                            
                            <div class="chat-header">
                                <h5 class="mb-0">
                                    <?php echo $other_user['company_name'] ?? $other_user['first_name'] . ' ' . $other_user['last_name']; ?>
                                </h5>
                                <small class="text-muted">
                                    <?php echo $other_user['role']; ?> • 
                                    <?php echo $conversation ? count($conversation) : 0; ?> messages
                                </small>
                            </div>

                            <div class="messages-area">
                                <div class="messages-list" id="messagesList">
                                    <?php if ($conversation): ?>
                                        <?php foreach ($conversation as $msg): ?>
                                            <div class="message <?php echo $msg['sender_id'] == $user_id ? 'sent' : 'received'; ?>">
                                                <div class="message-content">
                                                    <?php echo nl2br(htmlspecialchars($msg['message'])); ?>
                                                </div>
                                                <small class="message-time">
                                                    <?php echo date('M j, g:i A', strtotime($msg['created_at'])); ?>
                                                    <?php if ($msg['sender_id'] == $user_id && $msg['is_read']): ?>
                                                        <i class="fas fa-check-double text-success ms-1" title="Read"></i>
                                                    <?php endif; ?>
                                                </small>
                                            </div>
                                        <?php endforeach; ?>
                                    <?php else: ?>
                                        <div class="text-center py-5">
                                            <i class="fas fa-comments fa-2x text-muted mb-3"></i>
                                            <p class="text-muted">No messages in this conversation yet</p>
                                            <p class="text-muted">Start the conversation by sending a message below.</p>
                                        </div>
                                    <?php endif; ?>
                                </div>

                                <div class="message-form">
                                    <form method="POST" id="messageForm">
                                        <input type="hidden" name="receiver_id" value="<?php echo $conversation_user_id; ?>">
                                        <input type="hidden" name="message_type" value="General">
                                        <div class="input-group">
                                            <input type="text" name="message" class="form-control" placeholder="Type your message..." required>
                                            <button type="submit" name="send_message" class="btn btn-primary">
                                                <i class="fas fa-paper-plane"></i>
                                            </button>
                                        </div>
                                    </form>
                                </div>
                            </div>
                        <?php else: ?>
                            <div class="text-center py-5">
                                <i class="fas fa-comments fa-3x text-muted mb-3"></i>
                                <h4 class="text-muted">Select a conversation</h4>
                                <p class="text-muted">Choose a conversation from the list or start a new one.</p>
                                <button class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#newMessageModal">
                                    <i class="fas fa-plus me-2"></i>New Message
                                </button>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- New Message Modal -->
    <div class="modal fade" id="newMessageModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">New Message</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <div class="mb-3">
                            <label class="form-label">To</label>
                            <select name="receiver_id" class="form-select" required>
                                <option value="">Select a company...</option>
                                <?php foreach ($local_companies as $company): ?>
                                    <option value="<?php echo $company['id']; ?>">
                                        <?php echo $company['company_name'] ?? $company['first_name'] . ' ' . $company['last_name']; ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Subject</label>
                            <input type="text" name="subject" class="form-control" placeholder="Message subject...">
                        </div>
                        <div class="mb-3">
                            <label class="form-label">Message</label>
                            <textarea name="message" class="form-control" rows="4" placeholder="Type your message..." required></textarea>
                        </div>
                        <input type="hidden" name="message_type" value="General">
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" name="send_message" class="btn btn-primary">Send Message</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Auto-scroll to bottom of messages
        function scrollToBottom() {
            const messagesList = document.getElementById('messagesList');
            if (messagesList) {
                messagesList.scrollTop = messagesList.scrollHeight;
            }
        }

        // Scroll to bottom when page loads
        document.addEventListener('DOMContentLoaded', function() {
            scrollToBottom();
            
            // Auto-refresh messages every 10 seconds
            setInterval(function() {
                if (<?php echo $conversation_user_id ? 'true' : 'false'; ?>) {
                    location.reload();
                }
            }, 10000);
        });

        // Handle message form submission
        document.getElementById('messageForm')?.addEventListener('submit', function(e) {
            const messageInput = this.querySelector('input[name="message"]');
            if (messageInput.value.trim() === '') {
                e.preventDefault();
                alert('Please enter a message');
            }
        });
    </script>
</body>
</html>