<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isStore()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$state = $_SESSION['state'];
$lga = $_SESSION['lga'];
$message = '';

// Handle order actions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['cancel_order'])) {
        $order_id = intval($_POST['order_id']);
        
        // Verify order belongs to store
        $verify_stmt = $pdo->prepare("SELECT id FROM orders WHERE id = ? AND store_id = ? AND status = 'Pending'");
        $verify_stmt->execute([$order_id, $user_id]);
        
        if ($verify_stmt->fetch()) {
            $update_stmt = $pdo->prepare("UPDATE orders SET status = 'Cancelled' WHERE id = ?");
            if ($update_stmt->execute([$order_id])) {
                $message = '<div class="alert alert-success">Order cancelled successfully!</div>';
                
                // Restore product stock
                $order_stmt = $pdo->prepare("SELECT product_id, quantity FROM orders WHERE id = ?");
                $order_stmt->execute([$order_id]);
                $order = $order_stmt->fetch(PDO::FETCH_ASSOC);
                
                if ($order) {
                    $restore_stmt = $pdo->prepare("UPDATE products SET stock = stock + ? WHERE id = ?");
                    $restore_stmt->execute([$order['quantity'], $order['product_id']]);
                }
            } else {
                $message = '<div class="alert alert-danger">Failed to cancel order.</div>';
            }
        } else {
            $message = '<div class="alert alert-danger">Order not found or cannot be cancelled.</div>';
        }
    }
    
    if (isset($_POST['reorder'])) {
        $product_id = intval($_POST['product_id']);
        $quantity = intval($_POST['quantity']);
        
        // Get product details
        $product_stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
        $product_stmt->execute([$product_id]);
        $product = $product_stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($product && $quantity > 0 && $quantity <= $product['stock']) {
            // Initialize cart if not exists
            if (!isset($_SESSION['cart'])) {
                $_SESSION['cart'] = [];
            }
            
            // Add item to cart
            if (isset($_SESSION['cart'][$product_id])) {
                $_SESSION['cart'][$product_id]['quantity'] += $quantity;
            } else {
                $_SESSION['cart'][$product_id] = [
                    'product_id' => $product_id,
                    'name' => $product['name'],
                    'price' => $product['price'],
                    'quantity' => $quantity,
                    'company_id' => $product['company_id'],
                    'stock' => $product['stock']
                ];
            }
            
            $message = '<div class="alert alert-success">Product added to cart! <a href="store_cart.php" class="alert-link">View Cart</a></div>';
        } else {
            $message = '<div class="alert alert-danger">Invalid quantity or product out of stock.</div>';
        }
    }
}

// Get filter parameters
$status_filter = $_GET['status'] ?? '';
$payment_filter = $_GET['payment_status'] ?? '';
$date_from = $_GET['date_from'] ?? '';
$date_to = $_GET['date_to'] ?? '';
$search = $_GET['search'] ?? '';

// Build orders query with filters - FIXED: Removed company_name reference
$query = "
    SELECT o.*, 
           p.name as product_name,
           p.price as unit_price,
           u.first_name as company_first_name,
           u.last_name as company_last_name,
           u.phone as company_phone
    FROM orders o 
    JOIN products p ON o.product_id = p.id 
    JOIN users u ON o.company_id = u.id 
    WHERE o.store_id = ?
";

$params = [$user_id];

if ($status_filter) {
    $query .= " AND o.status = ?";
    $params[] = $status_filter;
}

if ($payment_filter) {
    $query .= " AND o.payment_status = ?";
    $params[] = $payment_filter;
}

if ($date_from) {
    $query .= " AND DATE(o.created_at) >= ?";
    $params[] = $date_from;
}

if ($date_to) {
    $query .= " AND DATE(o.created_at) <= ?";
    $params[] = $date_to;
}

if ($search) {
    $query .= " AND (p.name LIKE ? OR u.first_name LIKE ? OR u.last_name LIKE ?)";
    $search_term = "%$search%";
    $params[] = $search_term;
    $params[] = $search_term;
    $params[] = $search_term;
}

$query .= " ORDER BY o.created_at DESC";

$orders_stmt = $pdo->prepare($query);
$orders_stmt->execute($params);
$orders = $orders_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get order statistics
$stats_stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_orders,
        SUM(CASE WHEN status = 'Pending' THEN 1 ELSE 0 END) as pending_orders,
        SUM(CASE WHEN status = 'Processing' THEN 1 ELSE 0 END) as processing_orders,
        SUM(CASE WHEN status = 'Delivered' THEN 1 ELSE 0 END) as delivered_orders,
        SUM(CASE WHEN status = 'Cancelled' THEN 1 ELSE 0 END) as cancelled_orders,
        SUM(total_price) as total_spent,
        SUM(CASE WHEN payment_status = 'Paid' THEN total_price ELSE 0 END) as paid_amount
    FROM orders 
    WHERE store_id = ?
");
$stats_stmt->execute([$user_id]);
$stats = $stats_stmt->fetch(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Order History - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        .order-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 20px;
        }
        .order-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
        }
        .status-badge {
            font-size: 0.8em;
        }
        .order-actions {
            opacity: 0;
            transition: opacity 0.3s;
        }
        .order-card:hover .order-actions {
            opacity: 1;
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <?php include 'includes/store_sidebar.php'; ?>

    <div class="main-content">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-success me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-history me-2"></i>Order History
                </span>
                <a href="store_orders.php" class="btn btn-success">
                    <i class="fas fa-plus me-2"></i>New Order
                </a>
            </div>
        </nav>

        <?php echo $message; ?>

        <!-- Order Statistics -->
        <div class="row mb-4">
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-primary">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['total_orders'] ?? 0; ?></h4>
                        <small>Total Orders</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-warning">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['pending_orders'] ?? 0; ?></h4>
                        <small>Pending</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-info">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['processing_orders'] ?? 0; ?></h4>
                        <small>Processing</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-success">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['delivered_orders'] ?? 0; ?></h4>
                        <small>Delivered</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-danger">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0"><?php echo $stats['cancelled_orders'] ?? 0; ?></h4>
                        <small>Cancelled</small>
                    </div>
                </div>
            </div>
            <div class="col-xl-2 col-md-4 col-6 mb-3">
                <div class="card text-white bg-dark">
                    <div class="card-body text-center py-3">
                        <h4 class="mb-0">₦<?php echo number_format($stats['total_spent'] ?? 0, 0); ?></h4>
                        <small>Total Spent</small>
                    </div>
                </div>
            </div>
        </div>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <h5 class="card-title">Filter Orders</h5>
                <form method="GET" class="row g-3">
                    <div class="col-md-2">
                        <label class="form-label">Order Status</label>
                        <select name="status" class="form-select">
                            <option value="">All Statuses</option>
                            <option value="Pending" <?php echo $status_filter === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="Processing" <?php echo $status_filter === 'Processing' ? 'selected' : ''; ?>>Processing</option>
                            <option value="Shipped" <?php echo $status_filter === 'Shipped' ? 'selected' : ''; ?>>Shipped</option>
                            <option value="Delivered" <?php echo $status_filter === 'Delivered' ? 'selected' : ''; ?>>Delivered</option>
                            <option value="Cancelled" <?php echo $status_filter === 'Cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Payment Status</label>
                        <select name="payment_status" class="form-select">
                            <option value="">All Payments</option>
                            <option value="Pending" <?php echo $payment_filter === 'Pending' ? 'selected' : ''; ?>>Pending</option>
                            <option value="Paid" <?php echo $payment_filter === 'Paid' ? 'selected' : ''; ?>>Paid</option>
                            <option value="Failed" <?php echo $payment_filter === 'Failed' ? 'selected' : ''; ?>>Failed</option>
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">From Date</label>
                        <input type="date" name="date_from" class="form-control" value="<?php echo $date_from; ?>">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">To Date</label>
                        <input type="date" name="date_to" class="form-control" value="<?php echo $date_to; ?>">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Search</label>
                        <input type="text" name="search" class="form-control" placeholder="Product or company..." value="<?php echo $search; ?>">
                    </div>
                    <div class="col-md-1 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary w-100">Filter</button>
                    </div>
                    <div class="col-12">
                        <a href="store_order_history.php" class="btn btn-outline-secondary">Clear Filters</a>
                    </div>
                </form>
            </div>
        </div>

        <!-- Orders List -->
        <div class="card">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="card-title mb-0">Order History</h5>
                <span class="badge bg-primary"><?php echo count($orders); ?> orders</span>
            </div>
            <div class="card-body">
                <?php if ($orders): ?>
                    <div class="table-responsive">
                        <table class="table table-hover">
                            <thead>
                                <tr>
                                    <th>Order ID</th>
                                    <th>Product</th>
                                    <th>Company</th>
                                    <th>Quantity</th>
                                    <th>Unit Price</th>
                                    <th>Total Amount</th>
                                    <th>Order Status</th>
                                    <th>Payment Status</th>
                                    <th>Date Ordered</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($orders as $order): ?>
                                    <tr>
                                        <td>
                                            <strong>#<?php echo $order['id']; ?></strong>
                                        </td>
                                        <td>
                                            <strong><?php echo $order['product_name']; ?></strong>
                                        </td>
                                        <td>
                                            <!-- FIXED: Using first_name and last_name instead of company_name -->
                                            <?php echo $order['company_first_name'] . ' ' . $order['company_last_name']; ?>
                                            <br>
                                            <small class="text-muted"><?php echo $order['company_phone']; ?></small>
                                        </td>
                                        <td><?php echo $order['quantity']; ?></td>
                                        <td>₦<?php echo number_format($order['unit_price'], 2); ?></td>
                                        <td>
                                            <strong>₦<?php echo number_format($order['total_price'], 2); ?></strong>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php 
                                                switch($order['status']) {
                                                    case 'Delivered': echo 'bg-success'; break;
                                                    case 'Processing': echo 'bg-primary'; break;
                                                    case 'Shipped': echo 'bg-info'; break;
                                                    case 'Pending': echo 'bg-warning'; break;
                                                    case 'Cancelled': echo 'bg-danger'; break;
                                                    default: echo 'bg-secondary';
                                                }
                                                ?> status-badge">
                                                <?php echo $order['status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <span class="badge 
                                                <?php 
                                                switch($order['payment_status']) {
                                                    case 'Paid': echo 'bg-success'; break;
                                                    case 'Pending': echo 'bg-warning'; break;
                                                    case 'Failed': echo 'bg-danger'; break;
                                                    default: echo 'bg-secondary';
                                                }
                                                ?> status-badge">
                                                <?php echo $order['payment_status']; ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php echo date('M j, Y', strtotime($order['created_at'])); ?>
                                            <br>
                                            <small class="text-muted"><?php echo date('g:i A', strtotime($order['created_at'])); ?></small>
                                        </td>
                                        <td>
                                            <div class="order-actions">
                                                <!-- Reorder Button -->
                                                <?php if ($order['status'] !== 'Cancelled'): ?>
                                                    <form method="POST" class="d-inline">
                                                        <input type="hidden" name="product_id" value="<?php echo $order['product_id']; ?>">
                                                        <input type="hidden" name="quantity" value="<?php echo $order['quantity']; ?>">
                                                        <button type="submit" name="reorder" class="btn btn-sm btn-outline-success" title="Reorder">
                                                            <i class="fas fa-redo"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>

                                                <!-- Cancel Button (only for pending orders) -->
                                                <?php if ($order['status'] === 'Pending'): ?>
                                                    <form method="POST" class="d-inline" onsubmit="return confirm('Are you sure you want to cancel this order?')">
                                                        <input type="hidden" name="order_id" value="<?php echo $order['id']; ?>">
                                                        <button type="submit" name="cancel_order" class="btn btn-sm btn-outline-danger" title="Cancel Order">
                                                            <i class="fas fa-times"></i>
                                                        </button>
                                                    </form>
                                                <?php endif; ?>

                                                <!-- Message Company -->
                                                <a href="store_messages.php?user_id=<?php echo $order['company_id']; ?>" class="btn btn-sm btn-outline-info" title="Message Company">
                                                    <i class="fas fa-comment"></i>
                                                </a>

                                                <!-- View Details -->
                                                <button class="btn btn-sm btn-outline-primary" 
                                                        data-bs-toggle="modal" 
                                                        data-bs-target="#orderDetailsModal"
                                                        onclick="viewOrderDetails(<?php echo $order['id']; ?>, '<?php echo addslashes($order['product_name']); ?>', '<?php echo addslashes($order['company_first_name'] . ' ' . $order['company_last_name']); ?>', <?php echo $order['quantity']; ?>, <?php echo $order['unit_price']; ?>, <?php echo $order['total_price']; ?>, '<?php echo $order['status']; ?>', '<?php echo $order['payment_status']; ?>', '<?php echo $order['created_at']; ?>')">
                                                    <i class="fas fa-eye"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <div class="text-center py-5">
                        <i class="fas fa-shopping-cart fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Orders Found</h4>
                        <p class="text-muted">
                            <?php if ($status_filter || $payment_filter || $date_from || $date_to || $search): ?>
                                No orders match your search criteria.
                            <?php else: ?>
                                You haven't placed any orders yet.
                            <?php endif; ?>
                        </p>
                        <a href="store_orders.php" class="btn btn-success">
                            <i class="fas fa-plus me-2"></i>Place Your First Order
                        </a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Order Details Modal -->
    <div class="modal fade" id="orderDetailsModal" tabindex="-1">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Order Details - #<span id="modalOrderId"></span></h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h6>Product Information</h6>
                            <p><strong>Product:</strong> <span id="modalProductName"></span></p>
                            <p><strong>Quantity:</strong> <span id="modalQuantity"></span></p>
                            <p><strong>Unit Price:</strong> ₦<span id="modalUnitPrice"></span></p>
                            <p><strong>Total Amount:</strong> ₦<span id="modalTotalPrice"></span></p>
                        </div>
                        <div class="col-md-6">
                            <h6>Company Information</h6>
                            <p><strong>Company:</strong> <span id="modalCompanyName"></span></p>
                            <p><strong>Order Status:</strong> <span id="modalOrderStatus" class="badge"></span></p>
                            <p><strong>Payment Status:</strong> <span id="modalPaymentStatus" class="badge"></span></p>
                            <p><strong>Order Date:</strong> <span id="modalOrderDate"></span></p>
                        </div>
                    </div>
                    <div class="mt-4">
                        <h6>Order Timeline</h6>
                        <div class="timeline">
                            <div class="timeline-item">
                                <i class="fas fa-shopping-cart text-primary"></i>
                                <span>Order Placed</span>
                                <small id="modalOrderDateFull"></small>
                            </div>
                            <div class="timeline-item">
                                <i class="fas fa-cog text-warning"></i>
                                <span>Order Processing</span>
                                <small>Will update when processed</small>
                            </div>
                            <div class="timeline-item">
                                <i class="fas fa-shipping-fast text-info"></i>
                                <span>Order Shipped</span>
                                <small>Will update when shipped</small>
                            </div>
                            <div class="timeline-item">
                                <i class="fas fa-check-circle text-success"></i>
                                <span>Order Delivered</span>
                                <small>Will update when delivered</small>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <a href="#" id="modalMessageLink" class="btn btn-primary">
                        <i class="fas fa-comment me-2"></i>Message Company
                    </a>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle')?.addEventListener('click', function() {
            const sidebar = document.getElementById('sidebar');
            const mainContent = document.querySelector('.main-content');
            
            if (sidebar.style.marginLeft === '0px' || sidebar.style.marginLeft === '') {
                sidebar.style.marginLeft = '-250px';
                mainContent.style.marginLeft = '0';
            } else {
                sidebar.style.marginLeft = '0px';
                mainContent.style.marginLeft = '250px';
            }
        });

        // View order details function
        function viewOrderDetails(orderId, productName, companyName, quantity, unitPrice, totalPrice, orderStatus, paymentStatus, orderDate) {
            document.getElementById('modalOrderId').textContent = orderId;
            document.getElementById('modalProductName').textContent = productName;
            document.getElementById('modalCompanyName').textContent = companyName;
            document.getElementById('modalQuantity').textContent = quantity;
            document.getElementById('modalUnitPrice').textContent = unitPrice.toLocaleString();
            document.getElementById('modalTotalPrice').textContent = totalPrice.toLocaleString();
            
            // Set status badges
            const orderStatusBadge = document.getElementById('modalOrderStatus');
            orderStatusBadge.textContent = orderStatus;
            orderStatusBadge.className = 'badge ' + getStatusBadgeClass(orderStatus);
            
            const paymentStatusBadge = document.getElementById('modalPaymentStatus');
            paymentStatusBadge.textContent = paymentStatus;
            paymentStatusBadge.className = 'badge ' + getPaymentBadgeClass(paymentStatus);
            
            // Format dates
            const orderDateObj = new Date(orderDate);
            document.getElementById('modalOrderDate').textContent = orderDateObj.toLocaleDateString();
            document.getElementById('modalOrderDateFull').textContent = orderDateObj.toLocaleString();
            
            // Set message link
            document.getElementById('modalMessageLink').href = `store_messages.php?order_id=${orderId}`;
        }

        // Helper functions for badge classes
        function getStatusBadgeClass(status) {
            switch(status) {
                case 'Delivered': return 'bg-success';
                case 'Processing': return 'bg-primary';
                case 'Shipped': return 'bg-info';
                case 'Pending': return 'bg-warning';
                case 'Cancelled': return 'bg-danger';
                default: return 'bg-secondary';
            }
        }

        function getPaymentBadgeClass(status) {
            switch(status) {
                case 'Paid': return 'bg-success';
                case 'Pending': return 'bg-warning';
                case 'Failed': return 'bg-danger';
                default: return 'bg-secondary';
            }
        }

        // Auto-hide alerts after 5 seconds
        setTimeout(function() {
            const alerts = document.querySelectorAll('.alert');
            alerts.forEach(alert => {
                const bsAlert = new bootstrap.Alert(alert);
                bsAlert.close();
            });
        }, 5000);

        // Show order actions on hover
        document.querySelectorAll('tr').forEach(row => {
            row.addEventListener('mouseenter', function() {
                const actions = this.querySelector('.order-actions');
                if (actions) actions.style.opacity = '1';
            });
            row.addEventListener('mouseleave', function() {
                const actions = this.querySelector('.order-actions');
                if (actions) actions.style.opacity = '0';
            });
        });
    </script>

    <style>
        .timeline {
            position: relative;
            padding-left: 30px;
        }
        .timeline-item {
            position: relative;
            margin-bottom: 20px;
            padding-left: 30px;
        }
        .timeline-item i {
            position: absolute;
            left: -25px;
            top: 0;
            background: white;
            padding: 5px;
        }
        .timeline-item span {
            display: block;
            font-weight: 500;
        }
        .timeline-item small {
            color: #6c757d;
        }
    </style>
</body>
</html>