<?php
include 'includes/db_connect.php';
include 'includes/functions.php';

if (!isLoggedIn() || !isStore()) {
    redirect('login.php');
}

$user_id = $_SESSION['user_id'];
$state = $_SESSION['state'];
$lga = $_SESSION['lga'];

// Handle adding to cart
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_to_cart'])) {
    $product_id = intval($_POST['product_id']);
    $quantity = intval($_POST['quantity']);
    
    // Get product details
    $product_stmt = $pdo->prepare("SELECT * FROM products WHERE id = ?");
    $product_stmt->execute([$product_id]);
    $product = $product_stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($product && $quantity > 0 && $quantity <= $product['stock']) {
        // Initialize cart if not exists
        if (!isset($_SESSION['cart'])) {
            $_SESSION['cart'] = [];
        }
        
        // Add item to cart
        if (isset($_SESSION['cart'][$product_id])) {
            $_SESSION['cart'][$product_id]['quantity'] += $quantity;
        } else {
            $_SESSION['cart'][$product_id] = [
                'product_id' => $product_id,
                'name' => $product['name'],
                'price' => $product['price'],
                'quantity' => $quantity,
                'company_id' => $product['company_id'],
                'stock' => $product['stock']
            ];
        }
        
        $_SESSION['success'] = "Product added to cart!";
        redirect('store_orders.php');
    } else {
        $_SESSION['error'] = "Invalid quantity or product out of stock!";
    }
}

// Handle filters
$category_filter = $_GET['category'] ?? '';
$company_filter = $_GET['company'] ?? '';

// Build product query with filters
$query = "
    SELECT p.*, u.first_name, u.last_name 
    FROM products p 
    JOIN users u ON p.company_id = u.id 
    WHERE u.role = 'Company' AND u.state = ? AND u.lga = ? AND p.stock > 0
";

$params = [$state, $lga];

if ($category_filter) {
    $query .= " AND p.category = ?";
    $params[] = $category_filter;
}

if ($company_filter) {
    $query .= " AND u.id = ?";
    $params[] = $company_filter;
}

$query .= " ORDER BY p.created_at DESC";

$products_stmt = $pdo->prepare($query);
$products_stmt->execute($params);
$products = $products_stmt->fetchAll(PDO::FETCH_ASSOC);

// Get categories for filter
$categories_stmt = $pdo->prepare("SELECT DISTINCT category FROM products");
$categories_stmt->execute();
$all_categories = $categories_stmt->fetchAll(PDO::FETCH_ASSOC);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Make Order - MultiVendor System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <style>
        .product-card {
            border: none;
            border-radius: 10px;
            transition: all 0.3s;
            margin-bottom: 20px;
            height: 100%;
        }
        .product-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 10px 25px rgba(0,0,0,0.1);
        }
        .cart-badge {
            position: absolute;
            top: -5px;
            right: -5px;
        }
        .stock-badge {
            position: absolute;
            top: 10px;
            right: 10px;
        }
        .main-content {
            margin-left: 250px;
            padding: 20px;
            transition: all 0.3s;
        }
        @media (max-width: 768px) {
            .main-content {
                margin-left: 0;
            }
        }
    </style>
</head>
<body>
    <!-- Include the store sidebar -->
    <?php include 'includes/store_sidebar.php'; ?>

    <div class="main-content" id="mainContent">
        <!-- Top Navbar -->
        <nav class="navbar navbar-expand-lg navbar-light bg-white shadow-sm mb-4">
            <div class="container-fluid">
                <button class="btn btn-outline-success me-3 d-md-none" id="sidebarToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <span class="navbar-brand mb-0 h1">
                    <i class="fas fa-shopping-cart me-2"></i>Make Order
                </span>
                <div class="navbar-nav ms-auto">
                    <a href="store_cart.php" class="btn btn-success position-relative">
                        <i class="fas fa-shopping-cart me-2"></i>Cart
                        <?php if (isset($_SESSION['cart']) && count($_SESSION['cart']) > 0): ?>
                            <span class="position-absolute top-0 start-100 translate-middle badge rounded-pill bg-danger">
                                <?php echo count($_SESSION['cart']); ?>
                            </span>
                        <?php endif; ?>
                    </a>
                </div>
            </div>
        </nav>

        <?php if (isset($_SESSION['success'])): ?>
            <div class="alert alert-success alert-dismissible fade show">
                <?php echo $_SESSION['success']; unset($_SESSION['success']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <?php if (isset($_SESSION['error'])): ?>
            <div class="alert alert-danger alert-dismissible fade show">
                <?php echo $_SESSION['error']; unset($_SESSION['error']); ?>
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        <?php endif; ?>

        <!-- Filters -->
        <div class="card mb-4">
            <div class="card-body">
                <h5 class="card-title">Filter Products</h5>
                <form method="GET" class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Category</label>
                        <select name="category" class="form-select">
                            <option value="">All Categories</option>
                            <?php foreach ($all_categories as $category): ?>
                                <option value="<?php echo $category['category']; ?>" 
                                    <?php echo ($category_filter === $category['category']) ? 'selected' : ''; ?>>
                                    <?php echo $category['category']; ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Actions</label>
                        <div>
                            <button type="submit" class="btn btn-primary">Apply Filters</button>
                            <a href="store_orders.php" class="btn btn-outline-secondary">Clear</a>
                        </div>
                    </div>
                </form>
            </div>
        </div>

        <!-- Products Grid -->
        <div class="row">
            <?php if ($products): ?>
                <?php foreach ($products as $product): ?>
                    <div class="col-xl-3 col-lg-4 col-md-6 mb-4">
                        <div class="card product-card">
                            <div class="card-body d-flex flex-column">
                                <div class="position-relative">
                                    <?php if ($product['stock'] < 10): ?>
                                        <span class="badge bg-warning stock-badge">Low Stock</span>
                                    <?php endif; ?>
                                </div>
                                
                                <h5 class="card-title"><?php echo $product['name']; ?></h5>
                                <h6 class="card-subtitle mb-2 text-muted"><?php echo $product['category']; ?></h6>
                                
                                <p class="card-text flex-grow-1"><?php echo $product['description'] ?? 'No description available'; ?></p>
                                
                                <div class="mb-3">
                                    <strong class="text-primary">₦<?php echo number_format($product['price'], 2); ?></strong>
                                    <small class="text-muted"> per unit</small>
                                </div>
                                
                                <div class="mb-3">
                                    <small class="text-muted">Stock: <?php echo $product['stock']; ?> units</small>
                                </div>
                                
                                <div class="mb-2">
                                    <small class="text-muted">
                                        Supplier: <?php echo $product['first_name'] . ' ' . $product['last_name']; ?>
                                    </small>
                                </div>
                                
                                <form method="POST" class="add-to-cart-form mt-auto">
                                    <input type="hidden" name="product_id" value="<?php echo $product['id']; ?>">
                                    <div class="row g-2">
                                        <div class="col-6">
                                            <input type="number" name="quantity" class="form-control" 
                                                   value="1" min="1" max="<?php echo $product['stock']; ?>" 
                                                   required>
                                        </div>
                                        <div class="col-6">
                                            <button type="submit" name="add_to_cart" class="btn btn-success w-100">
                                                <i class="fas fa-cart-plus"></i> Add
                                            </button>
                                        </div>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div class="col-12">
                    <div class="text-center py-5">
                        <i class="fas fa-box-open fa-3x text-muted mb-3"></i>
                        <h4 class="text-muted">No Products Available</h4>
                        <p class="text-muted">No products found matching your criteria.</p>
                        <a href="store_orders.php" class="btn btn-primary">View All Products</a>
                    </div>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Mobile sidebar toggle
        document.getElementById('sidebarToggle').addEventListener('click', function() {
            const sidebar = document.getElementById('sidebar');
            sidebar.style.marginLeft = sidebar.style.marginLeft === '0px' ? '-250px' : '0px';
        });
    </script>
</body>
</html>